/**
 * This file is part of Totara Enterprise Extensions.
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * Totara Enterprise Extensions is provided only to Totara
 * Learning Solutions LTD's customers and partners, pursuant to
 * the terms and conditions of a separate agreement with Totara
 * Learning Solutions LTD or its affiliate.
 *
 * If you do not have an agreement with Totara Learning Solutions
 * LTD, you may not access, use, modify, or distribute this software.
 * Please contact [licensing@totara.com] for more information.
 *
 * @author Simon Chester <simon.chester@totara.com>
 * @module tui
 */

import { get, set } from '../object';

describe('get', () => {
  it('returns the value at path', () => {
    const obj = { a: [{ b: { c: 3 } }], d: 4, e: { f: { g: 5 } } };
    expect(get(obj, [])).toBe(obj);
    expect(get(obj, [''])).toBe(undefined);
    expect(get(obj, ['q'])).toBe(undefined);
    expect(get(obj, ['q', 'w'])).toBe(undefined);
    expect(get(obj, ['a', '0', 'b', 'c'])).toBe(3);
    expect(get(obj, ['a', 0, 'b', 'c'])).toBe(3);
    expect(get(obj, ['d'])).toBe(4);
    expect(get(obj, ['e', 'f'])).toBe(obj.e.f);
    expect(get(obj, ['e', 'f', 'g'])).toBe(5);
  });

  it('gets top level properties', () => {
    const obj = {
      foo: true,
      [1]: 2,
    };
    expect(get(obj, 'foo')).toBe(true);
    expect(get(obj, 1)).toBe(2);
  });

  it('gets nested properties', () => {
    const obj = {
      foo: { bar: true },
      [1]: { baz: [1, 10, 100] },
    };
    expect(get(obj, ['foo', 'bar'])).toBe(true);
    expect(get(obj, [1, 'baz', 2])).toBe(100);
  });

  it('avoids getting disallowed properties', () => {
    const obj = {};
    expect(get(obj, '__proto__')).toBeUndefined();
    expect(get(obj, ['__proto__'])).toBeUndefined();
    expect(get(obj, 'constructor')).toBeUndefined();
    expect(get({ constructor: 1 }, 'constructor')).toBe(1);
    expect(get(obj, ['constructor'])).toBeUndefined();
    expect(get(obj, ['constructor', 'prototype'])).toBeUndefined();
  });
});

describe('set', () => {
  it('sets the value at path', () => {
    const obj = {};
    set(obj, ['b', 'c'], 4);
    expect(obj.b.c).toBe(4);
    set(obj, ['d', '3'], 5);
    expect(obj.d[3]).toBe(5);
    expect(Array.isArray(obj.d)).toBe(true);
    set(obj, ['e', '3', 'f'], 6);
    expect(obj.e[3].f).toBe(6);
  });

  it('sets nested properties', () => {
    const obj = {};
    set(obj, ['foo', 'bar'], true);
    set(obj, [1, 'baz', 2], 1000);
    expect(obj.foo.bar).toBe(true);
    expect(obj[1].baz[2]).toBe(1000);
  });

  it('takes bare string and array paths', () => {
    const obj = {};
    set(obj, 'a', 1);
    expect(obj.a).toBe(1);

    const arr = [];
    set(arr, 1, 8);
    expect(arr[1]).toBe(8);
  });

  it('creates either arrays or objects depending on key type', () => {
    const obj = {};
    set(obj, ['foo', 2], 1000);
    set(obj, ['bar', '2'], 1000);
    set(obj, ['baz', 'z'], 1000);
    expect(obj.foo).toBeArray();
    expect(obj.bar).toBeArray();
    expect(obj.baz).toBeObject();
  });

  it('avoids setting disallowed properties', () => {
    const obj = {};

    // __proto__
    set(obj, '__proto__', { fail: 1 });
    set(obj, ['__proto__', 'fail'], 1);
    expect(obj.__proto__).not.toHaveProperty('fail');

    // constructor
    set(obj, 'constructor', { fail: 1 });
    set(obj, ['constructor', 'fail']);
    expect(obj.constructor).not.toHaveProperty('fail');
    // should still work if constructor is not a function
    const otherConstructor = { constructor: 1 };
    set(otherConstructor, 'constructor', 2);
    expect(otherConstructor.constructor).toBe(2);
  });
});
