<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * This file contains backup and restore output renderers
 *
 * @package   core_backup
 * @copyright 2010 Sam Hemelryk
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

/**
 * The primary renderer for the backup.
 *
 * Can be retrieved with the following code:
 * <?php
 * $renderer = $PAGE->get_renderer('core', 'backup');
 * ?>
 *
 * @package   core_backup
 * @copyright 2010 Sam Hemelryk
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class core_backup_renderer extends plugin_renderer_base {

    /**
     * Renderers a progress bar for the backup or restore given the items that make it up.
     *
     * @param array $items An array of items
     * @return string
     */
    public function progress_bar(array $items) {
        foreach ($items as &$item) {
            $text = $item['text'];
            unset($item['text']);
            if (array_key_exists('link', $item)) {
                $link = $item['link'];
                unset($item['link']);
                $item = html_writer::link($link, $text, $item);
            } else {
                $item = html_writer::tag('span', $text, $item);
            }
        }
        return html_writer::tag('div', join(get_separator(), $items), array('class' => 'backup_progress clearfix'));
    }

    /**
     * The backup and restore pages may display a log (if any) in a scrolling box.
     *
     * @param string $loghtml Log content in HTML format
     * @return string HTML content that shows the log
     */
    public function log_display($loghtml) {
        global $OUTPUT;
        $out = html_writer::start_div('backup_log');
        $out .= $OUTPUT->heading(get_string('backuplog', 'backup'));
        $out .= html_writer::start_div('backup_log_contents');
        $out .= $loghtml;
        $out .= html_writer::end_div();
        $out .= html_writer::end_div();
        return $out;
    }

    /**
     * Prints a dependency notification
     *
     * @param string $message
     * @return string
     */
    public function dependency_notification($message) {
        return html_writer::tag('div', $message, array('class' => 'notification dependencies_enforced'));
    }

    /**
     * Displays the details of a backup file
     *
     * @param stdClass $details
     * @param moodle_url $nextstageurl
     * @param stored_file|string $archive backup file
     * @param array $destinations destination options
     * @param string|null $currentdestination
     * @return string
     */
    public function backup_details($details, $nextstageurl, $archive, $destinations, $currentdestination) {
        $yestick = $this->output->pix_icon('i/valid', get_string('yes'));
        $notick = $this->output->pix_icon('i/invalid', get_string('no'));

        $html  = html_writer::start_tag('div', array('class' => 'backup-restore'));

        $html .= html_writer::start_tag('div', array('class' => 'backup-section'));
        $html .= $this->output->heading(get_string('backupdetails', 'backup'), 2, array('class' => 'header'));
        $html .= $this->backup_detail_pair(get_string('backuptype', 'backup'), get_string('backuptype'.$details->type, 'backup'));
        $html .= $this->backup_detail_pair(get_string('backupformat', 'backup'), get_string('backupformat'.$details->format, 'backup'));
        $html .= $this->backup_detail_pair(get_string('backupmode', 'backup'), get_string('backupmode'.$details->mode, 'backup'));
        $html .= $this->backup_detail_pair(get_string('backupdate', 'backup'), userdate($details->backup_date));
        if (!empty($details->totara_release)) {
            $html .= $this->backup_detail_pair(get_string('moodleversion', 'backup'),
                html_writer::tag('span', s($details->totara_release), array('class' => 'moodle_release')) .
                html_writer::tag('span', '[' . s($details->moodle_version) . ']', array('class' => 'moodle_version sub-detail')));
        } else {
            $html .= $this->backup_detail_pair(get_string('moodleversion', 'core'),
                html_writer::tag('span', s($details->moodle_release), array('class' => 'moodle_release')) .
                html_writer::tag('span', '[' . s($details->moodle_version) . ']', array('class' => 'moodle_version sub-detail')));
            $html .= $this->backup_detail_pair(get_string('backupversion', 'backup'),
                html_writer::tag('span', s($details->backup_release), array('class' => 'moodle_release')).
                html_writer::tag('span', '[' . s($details->backup_version) . ']', array('class' => 'moodle_version sub-detail')));
        }
        $html .= $this->backup_detail_pair(get_string('originalwwwroot', 'backup'),
            html_writer::tag('span', s($details->original_wwwroot), array('class' => 'originalwwwroot')).
            html_writer::tag('span', '[' . s($details->original_site_identifier_hash) . ']', array('class' => 'sitehash sub-detail')));
        if (!empty($details->include_file_references_to_external_content)) {
            $message = '';
            if (backup_general_helper::backup_is_samesite($details)) {
                $message = $yestick . ' ' . get_string('filereferencessamesite', 'backup');
            } else {
                $message = $notick . ' ' . get_string('filereferencesnotsamesite', 'backup');
            }
            $html .= $this->backup_detail_pair(get_string('includefilereferences', 'backup'), $message);
        }
        if ($archive) {
            if ($archive instanceof stored_file) {
                $filename = $archive->get_filename();
            } else {
                $filename = basename($archive);
            }
            $trusted = \backup_helper::is_trusted_backup($archive);
            $html .= $this->backup_detail_pair(get_string('file'), s($filename));
            $html .= $this->backup_detail_pair(get_string('backuptrusted', 'backup'), $trusted ? get_string('yes') : get_string('no'));
        }

        $html .= html_writer::end_tag('div');

        $html .= html_writer::start_tag('div', array('class' => 'backup-section settings-section'));
        $html .= $this->output->heading(get_string('backupsettings', 'backup'), 2, array('class' => 'header'));
        foreach ($details->root_settings as $label => $value) {
            if ($label == 'filename' or $label == 'user_files') {
                continue;
            }
            $html .= $this->backup_detail_pair(get_string('rootsetting'.str_replace('_', '', $label), 'backup'), $value ? $yestick : $notick);
        }
        $html .= html_writer::end_tag('div');

        if ($details->type === 'course') {
            $html .= html_writer::start_tag('div', array('class' => 'backup-section'));
            $html .= $this->output->heading(get_string('backupcoursedetails', 'backup'), 2, array('class' => 'header'));
            $html .= $this->backup_detail_pair(get_string('coursetitle', 'backup'), format_string($details->course->title));
            $html .= $this->backup_detail_pair(get_string('courseid', 'backup'), clean_param($details->course->courseid, PARAM_INT));

            // Warning users about front page backups.
            if ($details->original_course_format === 'site') {
                $html .= $this->backup_detail_pair(get_string('type_format', 'plugin'), get_string('sitecourseformatwarning', 'backup'));
            }
            $html .= html_writer::start_tag('div', array('class' => 'backup-sub-section'));
            $html .= $this->output->heading(get_string('backupcoursesections', 'backup'), 3, array('class' => 'subheader'));
            foreach ($details->sections as $key => $section) {
                $included = $key.'_included';
                $userinfo = $key.'_userinfo';
                if ($section->settings[$included] && $section->settings[$userinfo]) {
                    $value = get_string('sectionincanduser', 'backup');
                } else if ($section->settings[$included]) {
                    $value = get_string('sectioninc', 'backup');
                } else {
                    continue;
                }
                $html .= $this->backup_detail_pair(get_string('backupcoursesection', 'backup', format_string($section->title)), $value);
                $table = null;
                foreach ($details->activities as $activitykey => $activity) {
                    if ($activity->sectionid != $section->sectionid) {
                        continue;
                    }
                    if (empty($table)) {
                        $table = new html_table();
                        $table->head = array(get_string('module', 'backup'), get_string('title', 'backup'), get_string('userinfo', 'backup'));
                        $table->colclasses = array('modulename', 'moduletitle', 'userinfoincluded');
                        $table->align = array('left', 'left', 'center');
                        $table->attributes = array('class' => 'activitytable generaltable');
                        $table->data = array();
                    }
                    $name = get_string('pluginname', $activity->modulename);
                    $icon = new pix_icon('icon', $name, $activity->modulename, array('class' => 'iconlarge icon-pre'));
                    $table->data[] = array(
                        $this->output->render($icon).$name,
                        format_string($activity->title),
                        ($activity->settings[$activitykey.'_userinfo']) ? $yestick : $notick,
                    );
                }
                if (!empty($table)) {
                    $html .= $this->backup_detail_pair(get_string('sectionactivities', 'backup'), html_writer::table($table));
                }

            }
            $html .= html_writer::end_tag('div');
            $html .= html_writer::end_tag('div');
        }

        $html .= html_writer::start_tag('div', array('class' => 'backup-section'));
        $html .= $this->output->heading(get_string('restorestage2', 'backup'), 2, array('class' => 'header'));
        if ($destinations) {
            $html .= html_writer::start_tag('form', array('method' => 'post', 'action' => $nextstageurl->out_omit_querystring(),
                'class' => 'mform'));
            foreach ($nextstageurl->params() as $key => $value) {
                $html .= html_writer::empty_tag('input', array('type' => 'hidden', 'name' => $key, 'value' => $value));
            }

            if (!$currentdestination) {
                reset($destinations);
                $currentdestination = key($destinations);
            }
            foreach ($destinations as $value => $desc) {
                $attrs = array('type' => 'radio', 'name' => 'destination', 'value' => $value, 'id' => 'destination' . $value);
                if ($value == $currentdestination) {
                    $attrs['checked'] = 'checked';
                }
                $label = html_writer::label($desc, $attrs['id']);
                $html .= $this->backup_detail_pair('', html_writer::tag('input', $label, $attrs));
            }

            $html .= html_writer::start_tag('div', array('class' => 'fgroup'));
            $attrs = array('type' => 'submit', 'value' => get_string('next'), 'class' => 'proceedbutton', 'id' => 'id_submitbutton');
            $html .= html_writer::empty_tag('input', $attrs);
            $html .= html_writer::end_tag('div');
            $html .= html_writer::end_tag('form');

        } else {
            $html .= $this->output->notification(get_string('norestoreoptions', 'backup'));
        }
        $cancelurl = new moodle_url($this->page->url, array('cancel' => 1, 'sesskey' => sesskey()));
        $html .= $this->output->single_button($cancelurl, get_string('cancel'), 'post');

        $html .= html_writer::end_tag('div');

        $html .= html_writer::end_tag('div');

        return $html;
    }

    /**
     * Displays the general information about a backup file with non-standard format
     *
     * @param stdClass $details
     * @param moodle_url $nextstageurl
     * @param stored_file|string $archive backup file
     * @param array $destinations destination options
     * @param string|null $currentdestination
     * @return string
     */
    public function backup_details_nonstandard($details, $nextstageurl, $archive, $destinations, $currentdestination) {

        $html  = html_writer::start_tag('div', array('class' => 'backup-restore nonstandardformat'));
        $html .= html_writer::start_tag('div', array('class' => 'backup-section'));
        $html .= $this->output->heading(get_string('backupdetails', 'backup'), 2, 'header');
        $html .= $this->output->box(get_string('backupdetailsnonstandardinfo', 'backup'), 'noticebox');
        $html .= $this->backup_detail_pair(
            get_string('backupformat', 'backup'),
            get_string('backupformat' . $details->format, 'backup'));
        $html .= $this->backup_detail_pair(
            get_string('backuptype', 'backup'),
            get_string('backuptype' . $details->type, 'backup'));

        if ($archive) {
            if ($archive instanceof stored_file) {
                $filename = $archive->get_filename();
            } else {
                $filename = basename($archive);
            }
            $trusted = \backup_helper::is_trusted_backup($archive);
            $html .= $this->backup_detail_pair(get_string('file'), s($filename));
            $html .= $this->backup_detail_pair(get_string('backuptrusted', 'backup'), $trusted ? get_string('yes') : get_string('no'));
        }

        $html .= html_writer::end_tag('div');

        $html .= html_writer::start_tag('div', array('class' => 'backup-section'));
        $html .= $this->output->heading(get_string('restorestage2', 'backup'), 2, array('class' => 'header'));
        if ($destinations) {
            $html .= html_writer::start_tag('form', array('method' => 'post', 'action' => $nextstageurl->out_omit_querystring(),
                'class' => 'mform'));
            foreach ($nextstageurl->params() as $key => $value) {
                $html .= html_writer::empty_tag('input', array('type' => 'hidden', 'name' => $key, 'value' => $value));
            }

            if (!$currentdestination) {
                reset($destinations);
                $currentdestination = key($destinations);
            }
            foreach ($destinations as $value => $desc) {
                $attrs = array('type' => 'radio', 'name' => 'destination', 'value' => $value, 'id' => 'destination' . $value);
                if ($value == $currentdestination) {
                    $attrs['checked'] = 'checked';
                }
                $label = html_writer::label($desc, $attrs['id']);
                $html .= $this->backup_detail_pair('', html_writer::tag('input', $label, $attrs));
            }

            $html .= html_writer::start_tag('div', array('class' => 'fgroup'));
            $attrs = array('type' => 'submit', 'value' => get_string('next'), 'class' => 'proceedbutton', 'id' => 'id_submitbutton');
            $html .= html_writer::empty_tag('input', $attrs);
            $html .= html_writer::end_tag('div');
            $html .= html_writer::end_tag('form');

        } else {
            $html .= $this->output->notification(get_string('norestoreoptions', 'backup'));
        }
        $cancelurl = new moodle_url($this->page->url, array('cancel' => 1, 'sesskey' => sesskey()));
        $html .= $this->output->single_button($cancelurl, get_string('cancel'), 'post');
        $html .= html_writer::end_tag('div');

        $html .= html_writer::end_tag('div');

        return $html;
    }

    /**
     * Displays the general information about a backup file with unknown format
     *
     * @param moodle_url $nextstageurl URL to send user to
     * @return string HTML code to display
     */
    public function backup_details_unknown(moodle_url $nextstageurl) {

        $html  = html_writer::start_div('unknownformat');
        $html .= $this->output->heading(get_string('errorinvalidformat', 'backup'), 2);
        $html .= $this->output->notification(get_string('errorinvalidformatinfo', 'backup'), 'notifyproblem');
        $html .= $this->output->single_button($nextstageurl, get_string('continue'), 'post');
        $html .= html_writer::end_div();

        return $html;
    }

    /**
     * Displays a course selector for restore into new course.
     *
     * @param int $type backup type
     * @param moodle_url $nextstageurl
     * @param moodle_url $prevstageurl
     * @param restore_category_search $categories
     * @return string
     */
    public function course_selector_new($type, moodle_url $nextstageurl, moodle_url $prevstageurl, restore_category_search $categories) {
        // This is not a nice hack, but then this should have been done with normal forms!
        $missingdata = false;
        $targetid = optional_param('targetid', 0, PARAM_INT);
        $continue = optional_param('continue', 0, PARAM_BOOL);
        $stage    = optional_param('stage', null, PARAM_INT);
        if ($stage == restore_ui::STAGE_SETTINGS) {
            if ($continue and !$targetid) {
                $missingdata = true;
            }
        }

        $form = html_writer::start_tag('form', array('method' => 'post', 'action' => $nextstageurl->out_omit_querystring(),
            'class' => 'mform'));
        foreach ($nextstageurl->params() as $key => $value) {
            $form .= html_writer::empty_tag('input', array('type' => 'hidden', 'name' => $key, 'value' => $value));
        }

        $html  = html_writer::start_tag('div', array('class' => 'backup-course-selector backup-restore'));
        $html .= $form;
        $html .= html_writer::start_tag('div', array('class' => 'bcs-new-course backup-section'));
        $html .= $this->output->heading(get_string('restoretonewcourse', 'backup'), 2, array('class' => 'header'));
        $selectacategoryhtml = $this->backup_detail_pair(get_string('selectacategory', 'backup'), $this->render($categories));
        // Display the category selection as required if the form was submitted but this data was not supplied.
        if ($missingdata) {
            $html .= html_writer::span(get_string('required'), 'error');
            $html .= html_writer::start_tag('fieldset', array('class' => 'error'));
            $html .= $selectacategoryhtml;
            $html .= html_writer::end_tag('fieldset');
        } else {
            $html .= $selectacategoryhtml;
        }

        if ($categories->get_results()) {
            $html .= html_writer::start_tag('div', array('class' => 'fgroup'));
            $attrs = array('type' => 'submit', 'name' => 'continue', 'value' => get_string('next'), 'class' => 'proceedbutton', 'id' => 'id_submitbutton');
            $html .= html_writer::empty_tag('input', $attrs);
            $html .= html_writer::end_tag('div');
        }

        // Totara: Close the 'bcs-new-course backup-section' div, then the form.
        $html .= html_writer::end_tag('div');
        $html .= html_writer::end_tag('form');

        $html .= html_writer::start_tag('div', array('class' => 'buttons'));
        $html .= $this->output->single_button($prevstageurl, get_string('previous'), 'post');
        $cancelurl = new moodle_url($this->page->url, array('cancel' => 1, 'sesskey' => sesskey()));
        $html .= $this->output->single_button($cancelurl, get_string('cancel'), 'post');
        $html .= html_writer::end_tag('div');

        $html .= html_writer::end_tag('div');
        return $html;
    }

    /**
     * Displays a course selector for restore into new course.
     *
     * @param int $type backup type
     * @param moodle_url $nextstageurl
     * @param moodle_url $prevstageurl
     * @return string
     */
    public function course_selector_current($type, moodle_url $nextstageurl, moodle_url $prevstageurl) {
        $form = html_writer::start_tag('form', array('method' => 'post', 'action' => $nextstageurl->out_omit_querystring(),
            'class' => 'mform'));
        foreach ($nextstageurl->params() as $key => $value) {
            $form .= html_writer::empty_tag('input', array('type' => 'hidden', 'name' => $key, 'value' => $value));
        }

        $html  = html_writer::start_tag('div', array('class' => 'backup-course-selector backup-restore'));
        $html .= $form;
        $html .= html_writer::start_tag('div', array('class' => 'bcs-current-course backup-section'));
        $html .= $this->output->heading(get_string('restoretocurrentcourse', 'backup'), 2, array('class' => 'header'));

        if ($type == backup::TYPE_1COURSE) {
            $attrs = array('type' => 'checkbox', 'name' => 'deletedata', 'value' => 1, 'id' => 'deletedata');
            $label = html_writer::label(get_string('restoretocurrentcoursedeleting', 'backup'), $attrs['id']);
            $html .= $this->backup_detail_pair('', html_writer::tag('input', $label, $attrs));
        }

        $html .= html_writer::start_tag('div', array('class' => 'fgroup'));
        $attrs = array('type' => 'submit', 'name' => 'continue', 'value' => get_string('next'), 'class' => 'proceedbutton', 'id' => 'id_submitbutton');
        $html .= html_writer::empty_tag('input', $attrs);
        $html .= html_writer::end_tag('div');
        // Totara: Close the 'bcs-new-course backup-section' div, then the form.
        $html .= html_writer::end_tag('div');
        $html .= html_writer::end_tag('form');

        $html .= html_writer::start_tag('div', array('class' => 'buttons'));
        $html .= $this->output->single_button($prevstageurl, get_string('previous'), 'post');
        $cancelurl = new moodle_url($this->page->url, array('cancel' => 1, 'sesskey' => sesskey()));
        $html .= $this->output->single_button($cancelurl, get_string('cancel'), 'post');
        $html .= html_writer::end_tag('div');

        $html .= html_writer::end_tag('div');
        return $html;
    }

    /**
     * Displays a course selector for restore into new course.
     *
     * @param int $type backup type
     * @param moodle_url $nextstageurl
     * @param moodle_url $prevstageurl
     * @param restore_course_search $courses
     * @return string
     */
    public function course_selector_existing($type, moodle_url $nextstageurl, moodle_url $prevstageurl, restore_course_search $courses) {
        // This is not a nice hack, but then this should have been done with normal forms!
        $missingdata = false;
        $targetid = optional_param('targetid', 0, PARAM_INT);
        $continue = optional_param('continue', 0, PARAM_BOOL);
        $stage    = optional_param('stage', null, PARAM_INT);
        if ($stage == restore_ui::STAGE_SETTINGS) {
            if ($continue and !$targetid) {
                $missingdata = true;
            }
        }

        $form = html_writer::start_tag('form', array('method' => 'post', 'action' => $nextstageurl->out_omit_querystring(),
            'class' => 'mform'));
        foreach ($nextstageurl->params() as $key => $value) {
            $form .= html_writer::empty_tag('input', array('type' => 'hidden', 'name' => $key, 'value' => $value));
        }

        $html  = html_writer::start_tag('div', array('class' => 'backup-course-selector backup-restore'));
        $html .= $form;
        $html .= html_writer::start_tag('div', array('class' => 'bcs-existing-course backup-section'));
        $html .= $this->output->heading(get_string('restoretoexistingcourse', 'backup'), 2, array('class' => 'header'));

        $selectacategoryhtml = $this->backup_detail_pair(get_string('selectacategory', 'backup'), $this->render($courses));
        // Display the category selection as required if the form was submitted but this data was not supplied.
        if ($missingdata) {
            $html .= html_writer::span(get_string('required'), 'error');
            $html .= html_writer::start_tag('fieldset', array('class' => 'error'));
            $html .= $selectacategoryhtml;
            $html .= html_writer::end_tag('fieldset');
        } else {
            $html .= $selectacategoryhtml;
        }

        if ($type == backup::TYPE_1COURSE) {
            $attrs = array('type' => 'checkbox', 'name' => 'deletedata', 'value' => 1, 'id' => 'deletedata');
            $label = html_writer::label(get_string('restoretoexistingcoursedeleting', 'backup'), $attrs['id']);
            $html .= $this->backup_detail_pair('', html_writer::tag('input', $label, $attrs));
        }

        if ($courses->get_results()) {
            $html .= html_writer::start_tag('div', array('class' => 'fgroup'));
            $attrs = array('type' => 'submit', 'name' => 'continue', 'value' => get_string('next'), 'class' => 'proceedbutton', 'id' => 'id_submitbutton');
            $html .= html_writer::empty_tag('input', $attrs);
            $html .= html_writer::end_tag('div');
        }

        // Totara: Close the 'bcs-new-course backup-section' div, then the form.
        $html .= html_writer::end_tag('div');
        $html .= html_writer::end_tag('form');

        $html .= html_writer::start_tag('div', array('class' => 'buttons'));
        $html .= $this->output->single_button($prevstageurl, get_string('previous'), 'post');
        $cancelurl = new moodle_url($this->page->url, array('cancel' => 1, 'sesskey' => sesskey()));
        $html .= $this->output->single_button($cancelurl, get_string('cancel'), 'post');
        $html .= html_writer::end_tag('div');

        $html .= html_writer::end_tag('div');
        return $html;
    }

    /**
     * Displays the import course selector
     *
     * @param moodle_url $nextstageurl
     * @param import_course_search $courses
     * @return string
     */
    public function import_course_selector(moodle_url $nextstageurl, import_course_search $courses = null) {
        $html  = html_writer::start_tag('div', array('class' => 'import-course-selector backup-restore'));
        $html .= html_writer::start_tag('form', array('method' => 'post', 'action' => $nextstageurl->out_omit_querystring()));
        foreach ($nextstageurl->params() as $key => $value) {
            $html .= html_writer::empty_tag('input', array('type' => 'hidden', 'name' => $key, 'value' => $value));
        }
        // We only allow import adding for now. Enforce it here.
        $html .= html_writer::empty_tag('input', array('type' => 'hidden', 'name' => 'target', 'value' => backup::TARGET_CURRENT_ADDING));
        $html .= html_writer::start_tag('div', array('class' => 'ics-existing-course backup-section'));
        $html .= $this->output->heading(get_string('importdatafrom'), 2, array('class' => 'header'));
        $html .= $this->backup_detail_pair(get_string('selectacourse', 'backup'), $this->render($courses));
        $attrs = array('type' => 'submit', 'value' => get_string('continue'), 'class' => 'btn btn-primary');
        // Disable continue button if there are not courses.
        if ($courses->get_count() == 0) {
            $attrs['disabled'] = 'disabled';
        }
        $html .= $this->backup_detail_pair('', html_writer::empty_tag('input', $attrs));
        $html .= html_writer::end_tag('div');
        $html .= html_writer::end_tag('form');
        $html .= html_writer::end_tag('div');
        return $html;
    }

    /**
     * Creates a detailed pairing (key + value)
     *
     * @staticvar int $count
     * @param string $label
     * @param string $value
     * @return string
     */
    protected function backup_detail_pair($label, $value) {
        static $count = 0;
        $count ++;
        $html  = html_writer::start_tag('div', array('class' => 'detail-pair'));
        $html .= html_writer::tag('span', $label, array('class' => 'detail-pair-label'));
        $html .= html_writer::tag('div', $value, array('class' => 'detail-pair-value'));
        $html .= html_writer::end_tag('div');
        return $html;
    }

    /**
     * Created a detailed pairing with an input
     *
     * @param string $label
     * @param string $type
     * @param string $name
     * @param string $value
     * @param array $attributes
     * @param string|null $description
     * @return string
     */
    protected function backup_detail_input($label, $type, $name, $value, array $attributes = array(), $description = null) {
        if (!empty($description)) {
            $description = html_writer::tag('span', $description, array('class' => 'description'));
        } else {
            $description = '';
        }
        return $this->backup_detail_pair(
            $label,
            html_writer::empty_tag('input', $attributes + array('name' => $name, 'type' => $type, 'value' => $value)) . $description
        );
    }

    /**
     * Creates a detailed pairing with a select
     *
     * @param string $label
     * @param string $name
     * @param array $options
     * @param string $selected
     * @param bool $nothing
     * @param array $attributes
     * @param string|null $description
     * @return string
     */
    protected function backup_detail_select($label, $name, $options, $selected = '', $nothing = false, array $attributes = array(), $description = null) {
        if (!empty ($description)) {
            $description = html_writer::tag('span', $description, array('class' => 'description'));
        } else {
            $description = '';
        }
        return $this->backup_detail_pair($label, html_writer::select($options, $name, $selected, false, $attributes).$description);
    }

    /**
     * Displays precheck notices
     *
     * @param array $results
     * @return string
     */
    public function precheck_notices($results) {
        $output = html_writer::start_tag('div', array('class' => 'restore-precheck-notices'));
        if (array_key_exists('errors', $results)) {
            foreach ($results['errors'] as $error) {
                $output .= $this->output->notification($error);
            }
        }
        if (array_key_exists('warnings', $results)) {
            foreach ($results['warnings'] as $warning) {
                $output .= $this->output->notification($warning, 'notifyproblem');
            }
        }
        return $output.html_writer::end_tag('div');
    }

    /**
     * Displays substage buttons
     *
     * @param bool $haserrors
     * @return string
     */
    public function substage_buttons($haserrors) {
        $output  = html_writer::start_tag('div', array('continuebutton'));
        if (!$haserrors) {
            $attrs = array('type' => 'submit', 'value' => get_string('continue'), 'class' => 'btn btn-primary');
            $output .= html_writer::empty_tag('input', $attrs);
        }
        $attrs = array('type' => 'submit', 'name' => 'cancel', 'value' => get_string('cancel'), 'class' => 'btn btn-secondary');
        $output .= html_writer::empty_tag('input', $attrs);
        $output .= html_writer::end_tag('div');
        return $output;
    }

    /**
     * Displays a role mapping interface
     *
     * @param array $rolemappings
     * @param array $roles
     * @return string
     */
    public function role_mappings($rolemappings, $roles) {
        $roles[0] = get_string('none');
        $output  = html_writer::start_tag('div', array('class' => 'restore-rolemappings'));
        $output .= $this->output->heading(get_string('restorerolemappings', 'backup'), 2);
        foreach ($rolemappings as $id => $mapping) {
            $label = $mapping->name;
            $name = 'mapping'.$id;
            $selected = $mapping->targetroleid;
            $output .= $this->backup_detail_select($label, $name, $roles, $mapping->targetroleid, false, array(), $mapping->description);
        }
        $output .= html_writer::end_tag('div');
        return $output;
    }

    /**
     * Displays a continue button
     *
     * @param string|moodle_url $url
     * @param string $method
     * @return string
     */
    public function continue_button($url, $method = 'post', $sesskey = true) {
        if (!($url instanceof moodle_url)) {
            $url = new moodle_url($url);
        }
        if ($method != 'post') {
            $method = 'get';
        }
        if ($sesskey) {
            $url->param('sesskey', sesskey());
        }
        $button = new single_button($url, get_string('continue'), $method, true);
        $button->class = 'continuebutton';
        return $this->render($button);
    }

    /**
     * Print an external automated backup files tree
     * @param array $options
     * @return string
     */
    public function backup_external_files_viewer(array $options) {
        $files = new backup_external_files_viewer($options);
        return $this->render($files);
    }

    /**
     * Print a backup files tree
     * @param array $options
     * @return string
     */
    public function backup_files_viewer(array $options) {
        $files = new backup_files_viewer($options);
        return $this->render($files);
    }

    /**
     * Displays a backup files viewer
     *
     * @param backup_files_viewer $viewer
     * @return string
     */
    public function render_backup_files_viewer(backup_files_viewer $viewer) {
        $files = $viewer->files;

        $table = new html_table();
        $table->attributes['class'] = 'backup-files-table generaltable';
        $table->head = array(get_string('filename', 'backup'), get_string('time'), get_string('size'), get_string('backuptrusted', 'backup'), get_string('actions'));
        $table->size = array('40%', '20%', '10%', '5%', '25%');
        $table->data = array();

        foreach ($files as $file) {
            if ($file->is_directory()) {
                continue;
            }
            $filename = s(ltrim($file->get_filepath() . $file->get_filename(), '/'));
            if ($viewer->candownload) {
                $fileurl = moodle_url::make_pluginfile_url(
                    $file->get_contextid(),
                    $file->get_component(),
                    $file->get_filearea(),
                    null,
                    $file->get_filepath(),
                    $file->get_filename(),
                    true
                );
                $filename = html_writer::link($fileurl, $filename);
            }

            $trusted = \backup_helper::is_trusted_backup($file);

            $data = array(
                $filename,
                userdate($file->get_timemodified()),
                display_size($file->get_filesize()),
                $trusted ? get_string('yes') : get_string('no'),
            );

            $actions = array();
            if ($viewer->canrestore and ($trusted or $viewer->allowuntrusted) and in_array($file->get_mimetype(), $viewer->mimetypes)) {
                $params = array('contextid' => $viewer->currentcontext->id, 'backupfileid' => $file->get_id(), 'sesskey' => sesskey());
                $url = new moodle_url('/backup/restore.php', $params);
                $restorebutton = new single_button($url, get_string('restore'), 'post', true);
                $actions[] = $this->render($restorebutton);
            }
            if ($viewer->candelete) {
                $params = array('contextid' => $viewer->currentcontext->id, 'deletefileid' => $file->get_id(), 'sesskey' => sesskey());
                $url = new moodle_url('/backup/restorefile.php', $params);
                $deletebutton = new single_button($url, get_string('delete'), 'post', false);
                $deletebutton->add_confirm_action(get_string('confirmbackupdelete', 'backup', s($file->get_filename())));
                $actions[] = $this->render($deletebutton);
            }
            if ($actions) {
                $data[] = '<span class="buttons">' . implode(' ', $actions) . '</span>';
            } else {
                $data[] = '';
            }

            $table->data[] = $data;
        }
        $html = html_writer::table($table);

        if ($viewer->canmanage) {
            $html .= $this->output->single_button(
                new moodle_url('/backup/backupfilesedit.php', array(
                        'contextid' => $viewer->currentcontext->id,
                        'filearea' => $viewer->filearea,
                        'component' => $viewer->component)
                ),
                get_string('managefiles', 'backup'),
                'post'
            );
        }

        return $html;
    }

    /**
     * Displays an external backup files viewer
     *
     * @param backup_external_files_viewer $viewer
     * @return string
     */
    public function render_backup_external_files_viewer(backup_external_files_viewer $viewer) {
        $files = $viewer->files;

        $table = new html_table();
        $table->attributes['class'] = 'backup-files-table generaltable';
        $table->head = array(get_string('filename', 'backup'), get_string('size'), get_string('backuptrusted', 'backup'), get_string('actions'));
        $table->size = array('60%', '10%', '5%', '25%');
        $table->data = array();

        foreach ($files as $file) {
            $filename = s($file->filename);
            $trusted = $file->trusted;

            $data = array(
                $filename,
                display_size($file->size),
                $trusted ? get_string('yes') : get_string('no'),
            );

            $actions = array();
            if ($viewer->canrestore and ($trusted or $viewer->allowuntrusted)) {
                $params = array('contextid' => $viewer->currentcontext->id, 'externalfilename' => $filename, 'sesskey' => sesskey());
                $url = new moodle_url('/backup/restore.php', $params);
                $restorebutton = new single_button($url, get_string('restore'), 'post', true);
                $actions[] = $this->render($restorebutton);
            }
            if ($actions) {
                $data[] = '<span class="buttons">' . implode(' ', $actions) . '</span>';
            } else {
                $data[] = '';
            }

            $table->data[] = $data;
        }
        $html = html_writer::table($table);

        return $html;
    }

    /**
     * Renders a restore course search object
     *
     * @param restore_course_search $component
     * @return string
     */
    public function render_restore_course_search(restore_course_search $component) {
        $url = $component->get_url();

        $output = html_writer::start_tag('div', array('class' => 'restore-course-search form-inline mb-3'));
        $output .= html_writer::start_tag('div', array('class' => 'rcs-results'));

        $table = new html_table();
        $table->head = array('', get_string('shortnamecourse'), get_string('fullnamecourse'));
        $table->data = array();
        if ($component->get_count() !== 0) {
            foreach ($component->get_results() as $course) {
                $row = new html_table_row();
                $row->attributes['class'] = 'rcs-course';
                if (!$course->visible) {
                    $row->attributes['class'] .= ' dimmed';
                }
                $id = 'id_restore_cs_target_' . $course->id;
                $row->cells = array(
                    html_writer::empty_tag('input', array('type' => 'radio', 'name' => 'targetid', 'value' => $course->id, 'id' => $id)),
                    html_writer::label(format_string($course->shortname, true, array('context' => context_course::instance($course->id))), $id),
                    format_string($course->fullname, true, array('context' => context_course::instance($course->id)))
                );
                $table->data[] = $row;
            }
            if ($component->has_more_results()) {
                $cell = new html_table_cell(get_string('moreresults', 'backup'));
                $cell->colspan = 3;
                $cell->attributes['class'] = 'notifyproblem';
                $row = new html_table_row(array($cell));
                $row->attributes['class'] = 'rcs-course';
                $table->data[] = $row;
            }
        } else {
            $cell = new html_table_cell(get_string('nomatchingcourses', 'backup'));
            $cell->colspan = 3;
            $cell->attributes['class'] = 'notifyproblem';
            $row = new html_table_row(array($cell));
            $row->attributes['class'] = 'rcs-course';
            $table->data[] = $row;
        }
        $output .= html_writer::table($table);
        $output .= html_writer::end_tag('div');

        $output .= html_writer::start_tag('div', array('class' => 'rcs-search'));
        $attrs = array(
            'type' => 'text',
            'name' => restore_course_search::$VAR_SEARCH,
            'value' => $component->get_search(),
            'class' => 'form-control'
        );
        $output .= html_writer::empty_tag('input', $attrs);
        $attrs = array(
            'type' => 'submit',
            'name' => 'searchcourses',
            'value' => get_string('search'),
            'class' => 'btn btn-secondary'
        );
        $output .= html_writer::empty_tag('input', $attrs);
        $output .= html_writer::end_tag('div');

        $output .= html_writer::end_tag('div');
        return $output;
    }

    /**
     * Renders an import course search object
     *
     * @param import_course_search $component
     * @return string
     */
    public function render_import_course_search(import_course_search $component) {
        $url = $component->get_url();

        $countsearchresults = $component->get_count();
        $countstr = get_string('totalcoursesearchresults', 'backup', $countsearchresults);
        if ($component->has_more_results()) {
            $countstr = get_string('morecoursesearchresults', 'backup', $countsearchresults);
        }

        $output = html_writer::start_tag('div', array('class' => 'import-course-search'));
        $output .= html_writer::tag('div', $countstr, array('class' => 'ics-totalresults'));
        $output .= html_writer::start_tag('div', array('class' => 'ics-results'));

        if ($countsearchresults === 0)  {
            $output .= get_string('nomatchingcoursesforsearch', 'backup', $component->get_search());
        } else {
            // Output table with the results.
            $table = new html_table();
            $table->head = array('', get_string('shortnamecourse'), get_string('fullnamecourse'));
            $table->data = array();
            foreach ($component->get_results() as $course) {
                $row = new html_table_row();
                $row->attributes['class'] = 'ics-course';
                if (!$course->visible) {
                    $row->attributes['class'] .= ' dimmed';
                }
                $row->cells = array(
                    html_writer::empty_tag('input', array('type' => 'radio', 'name' => 'importid', 'value' => $course->id, 'aria-label' => get_string('importdatafrom', 'backup', $course->fullname))),
                    format_string($course->shortname, true, array('context' => context_course::instance($course->id))),
                    format_string($course->fullname, true, array('context' => context_course::instance($course->id)))
                );
                $table->data[] = $row;
            }
            if ($component->has_more_results()) {
                $cell = new html_table_cell(get_string('moreresults', 'backup'));
                $cell->colspan = 3;
                $cell->attributes['class'] = 'notifyproblem';
                $row = new html_table_row(array($cell));
                $row->attributes['class'] = 'rcs-course';
                $table->data[] = $row;
            }
            $output .= html_writer::table($table);
        }

        $output .= html_writer::end_tag('div');

        $output .= html_writer::start_tag('div', array('class' => 'ics-search form-inline'));
        $attrs = array(
            'type' => 'text',
            'name' => restore_course_search::$VAR_SEARCH,
            'value' => $component->get_search(),
            'class' => 'form-control');
        $output .= html_writer::empty_tag('input', $attrs);
        $attrs = array(
            'type' => 'submit',
            'name' => 'searchcourses',
            'value' => get_string('search'),
            'class' => 'btn btn-secondary ml-1'
        );
        $output .= html_writer::empty_tag('input', $attrs);
        $attrs = array(
            'type' => 'submit',
            'name' => 'clearsearch',
            'value' => get_string('clear'),
            'class' => 'btn btn-secondary ml-1'
        );
        $output .= html_writer::empty_tag('input', $attrs);
        $output .= html_writer::end_tag('div');

        $output .= html_writer::end_tag('div');
        return $output;
    }

    /**
     * Renders a restore category search object
     *
     * @param restore_category_search $component
     * @return string
     */
    public function render_restore_category_search(restore_category_search $component) {
        $url = $component->get_url();

        $output = html_writer::start_tag('div', array('class' => 'restore-course-search form-inline mb-3'));
        $output .= html_writer::start_tag('div', array('class' => 'rcs-results'));

        $table = new html_table();
        $table->head = array('', get_string('name'), get_string('description'));
        $table->data = array();

        if ($component->get_count() !== 0) {
            foreach ($component->get_results() as $category) {
                $row = new html_table_row();
                $row->attributes['class'] = 'rcs-course';
                if (!$category->visible) {
                    $row->attributes['class'] .= ' dimmed';
                }
                $context = context_coursecat::instance($category->id);
                $id = 'id_restore_cats_target_' . $category->id;
                $row->cells = array(
                    html_writer::empty_tag('input', array('type' => 'radio', 'name' => 'targetid', 'value' => $category->id, 'id' => $id)),
                    html_writer::label(format_string($category->name, true, array('context' => context_coursecat::instance($category->id))), $id),
                    format_text(file_rewrite_pluginfile_urls($category->description, 'pluginfile.php', $context->id,
                        'coursecat', 'description', null), $category->descriptionformat, array('overflowdiv' => true))
                );
                $table->data[] = $row;
            }
            if ($component->has_more_results()) {
                $cell = new html_table_cell(get_string('moreresults', 'backup'));
                $cell->attributes['class'] = 'notifyproblem';
                $cell->colspan = 3;
                $row = new html_table_row(array($cell));
                $row->attributes['class'] = 'rcs-course';
                $table->data[] = $row;
            }
        } else {
            $cell = new html_table_cell(get_string('nomatchingcourses', 'backup'));
            $cell->colspan = 3;
            $cell->attributes['class'] = 'notifyproblem';
            $row = new html_table_row(array($cell));
            $row->attributes['class'] = 'rcs-course';
            $table->data[] = $row;
        }
        $output .= html_writer::table($table);
        $output .= html_writer::end_tag('div');

        $output .= html_writer::start_tag('div', array('class' => 'rcs-search'));
        $attrs = array(
            'type' => 'text',
            'name' => restore_category_search::$VAR_SEARCH,
            'value' => $component->get_search(),
            'class' => 'form-control'
        );
        $output .= html_writer::empty_tag('input', $attrs);
        $attrs = array(
            'type' => 'submit',
            'name' => 'searchcourses',
            'value' => get_string('search'),
            'class' => 'btn btn-secondary'
        );
        $output .= html_writer::empty_tag('input', $attrs);
        $output .= html_writer::end_tag('div');

        $output .= html_writer::end_tag('div');
        return $output;
    }
}

/**
 * Data structure representing backup files viewer
 *
 * @copyright 2010 Dongsheng Cai
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @since     Moodle 2.0
 */
class backup_files_viewer implements renderable {

    /**
     * @var array
     */
    public $files;

    /**
     * @var context
     */
    public $filecontext;

    /**
     * @var string
     */
    public $component;

    /**
     * @var string
     */
    public $filearea;

    /**
     * @var context
     */
    public $currentcontext;

    /**
     * @var bool
     */
    public $candownload;

    /**
     * @var bool
     */
    public $canrestore;

    /**
     * @var bool
     */
    public $canmanage;

    /**
     * @var bool
     */
    public $candelete;

    /**
     * @var bool
     */
    public $allowuntrusted;

    /**
     * @var string[]
     */
    public $mimetypes;

    /**
     * Constructor of backup_files_viewer class
     * @param array $options
     */
    public function __construct(array $options) {
        $fs = get_file_storage();
        $this->currentcontext = $options['currentcontext'];
        $this->filecontext    = $options['filecontext'];
        $this->component      = $options['component'];
        $this->filearea       = $options['filearea'];
        $this->candownload    = isset($options['candownload']) ? (bool)$options['candownload'] : false;
        $this->canrestore     = isset($options['canrestore']) ? (bool)$options['canrestore'] : false;
        $this->canmanage      = isset($options['canmanage']) ? (bool)$options['canmanage'] : false;
        $this->candelete      = isset($options['candelete']) ? (bool)$options['candelete'] : false;
        $this->allowuntrusted = isset($options['allowuntrusted']) ? (bool)$options['allowuntrusted'] : false;
        $this->mimetypes      = restore_ui_stage::get_allowed_mimetypes();
        $this->files = $fs->get_area_files($this->filecontext->id, $this->component, $this->filearea, false, 'timecreated DESC');
    }
}

/**
 * Data structure representing external automated backup files viewer
 *
 * @author Petr Skoda <petr.skoda@totaralearning.com>
 * @since Totara 11
 */
class backup_external_files_viewer implements renderable {

    /**
     * @var array
     */
    public $files;

    /**
     * @var context
     */
    public $currentcontext;

    /**
     * @var bool
     */
    public $canrestore;

    /**
     * @var bool
     */
    public $allowuntrusted;

    /**
     * Constructor of backup_files_viewer class
     * @param array $options
     */
    public function __construct(array $options) {
        $this->currentcontext = $options['currentcontext'];
        $this->canrestore     = isset($options['canrestore']) ? (bool)$options['canrestore'] : false;
        $this->allowuntrusted = isset($options['allowuntrusted']) ? (bool)$options['allowuntrusted'] : false;

        $autodestination = $options['autodestination'];
        $courseid = $this->currentcontext->instanceid;

        $totararegex = \backup_cron_automated_helper::get_external_file_regex($courseid, false);
        $oldregex = \backup_cron_automated_helper::get_external_file_regex($courseid, true);

        $backupfiles = array();
        $oldbackupfiles = array();
        foreach (scandir($autodestination) as $filename) {
            if (preg_match($totararegex, $filename)) {
                // Do not hash the files here, it would be too slow - nobody should be modifying these files!
                $backupfiles[$filename] = (object)array('filename' => $filename, 'size' => filesize($autodestination . '/' . $filename), 'trusted' => true);
                continue;
            }
            if (preg_match($oldregex, $filename)) {
                // Do not hash the files here, it would be too slow - nobody should be creating these files now.
                $oldbackupfiles[$filename] = (object)array('filename' => $filename, 'size' => filesize($autodestination . '/' . $filename), 'trusted' => false);
                continue;
            }
        }
        krsort($backupfiles);
        krsort($oldbackupfiles);

        $this->files = array_merge(array_values($backupfiles), array_values($oldbackupfiles));
    }
}
