<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Matthias Bonk <matthias.bonk@totara.com>
 * @package core_badges
 */

use core\orm\query\builder;
use core_badges\backpack_api;
use core_badges\helper\encryption_helper;
use core_phpunit\testcase;

global $CFG;
require_once($CFG->libdir . '/badgeslib.php');

class core_badges_backpack_api_test extends testcase {

    public function test_instantiate_using_site_backpack_with_unencrypted_password(): void {
        $builder = builder::table('badge_external_backpack');

        $site_backpack = $builder->get()->first();

        // Update with an unencrypted password.
        $builder
            ->where('id', $site_backpack->id)
            ->update(['password' => 'clear_text_password']);

        // Refresh
        $site_backpack = $builder->find($site_backpack->id);

        $backpack_api = new backpack_api($site_backpack);
        $this->assert_password_property($backpack_api, 'clear_text_password');
    }

    public function test_instantiate_using_site_backpack_with_encrypted_password(): void {
        $builder = builder::table('badge_external_backpack');

        $site_backpack = $builder->get()->first();

        // Update the record with an encrypted password.
        $encrypted_password = encryption_helper::encrypt_string(
            'secret_password',
            $site_backpack->id,
            encryption_helper::ITEM_TYPE_EXTERNAL
        );
        $builder
            ->where('id', $site_backpack->id)
            ->update(['password' => $encrypted_password]);

        // Refresh
        $site_backpack = $builder->find($site_backpack->id);

        $backpack_api = new backpack_api($site_backpack);
        $this->assert_password_property($backpack_api, 'secret_password');
    }

    public function test_instantiate_using_user_backpack_with_unencrypted_password(): void {
        // Still need a site backpack for instantiation.
        $site_backpack = builder::table('badge_external_backpack')->get()->first();

        $user_backpack = (object)[
            'id' => 123,
            'externalbackpackid' => $site_backpack->id,
            'email' => 'test@example.com',
            'password' => 'user_backpack_clear_text_password',
        ];

        $backpack_api = new backpack_api($site_backpack, $user_backpack);
        $this->assert_password_property($backpack_api, 'user_backpack_clear_text_password');
    }

    public function test_instantiate_using_user_backpack_with_encrypted_password(): void {
        // Still need a site backpack for instantiation.
        $site_backpack = builder::table('badge_external_backpack')->get()->first();

        $encrypted_password = encryption_helper::encrypt_string(
            'user_backpack_secret_password',
            123,
            encryption_helper::ITEM_TYPE_USER
        );

        $user_backpack = (object)[
            'id' => 123,
            'externalbackpackid' => $site_backpack->id,
            'email' => 'test@example.com',
            'password' => $encrypted_password,
        ];

        $backpack_api = new backpack_api($site_backpack, $user_backpack);
        $this->assert_password_property($backpack_api, 'user_backpack_secret_password');
    }

    /**
     * Instantiation doesn't require 'id' property for the user backpack, but we need it for decryption.
     * So we expect the password to be left in clear text, even when an encrypted one is passed in.
     * This shouldn't happen in real life. Just making sure nothing breaks.
     */
    public function test_instantiate_using_user_backpack_without_id(): void {
        // Still need a site backpack for instantiation.
        $site_backpack = builder::table('badge_external_backpack')->get()->first();

        $encrypted_password = encryption_helper::encrypt_string(
            'user_backpack_secret_password',
            123,
            encryption_helper::ITEM_TYPE_USER
        );

        $user_backpack = (object)[
            'externalbackpackid' => $site_backpack->id,
            'email' => 'test@example.com',
            'password' => $encrypted_password,
        ];

        $backpack_api = new backpack_api($site_backpack, $user_backpack);
        $this->assert_password_property($backpack_api, $encrypted_password);
    }

    /**
     * @param backpack_api $backpack_api
     * @param string $expected_password
     * @return void
     */
    private function assert_password_property(backpack_api $backpack_api, string $expected_password): void {
        $reflection = new ReflectionClass($backpack_api);
        $property = $reflection->getProperty('password');
        $property->setAccessible(true);
        $value = $property->getValue($backpack_api);

        $this->assertSame($expected_password, $value);
    }


}