<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Unit tests for badges
 *
 * @package    core
 * @subpackage badges
 * @copyright  2013 onwards Totara Learning Solutions Ltd {@link http://www.totaralms.com/}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @author     Yuliya Bozhko <yuliya.bozhko@totaralms.com>
 */

defined('MOODLE_INTERNAL') || die();

use core\orm\query\builder;
use core_badges\helper\encryption_helper;
use totara_program\assignments\assignments;
use totara_program\program;
use core\task\manager;

global $CFG;
require_once($CFG->libdir . '/badgeslib.php');
require_once($CFG->dirroot . '/badges/lib.php');

class core_badges_badgeslib_test extends \core_phpunit\testcase {
    protected $badgeid;
    protected $course;
    protected $user;
    protected $module;
    protected $coursebadge;
    protected $assertion;

    /** @var $assertion2 to define json format for Open badge version 2 */
    protected $assertion2;

    protected function tearDown(): void {
        $this->badgeid = null;
        $this->course = null;
        $this->user = null;
        $this->module = null;
        $this->coursebadge = null;
        $this->assertion = null;
        $this->assertion2 = null;
        parent::tearDown();
    }

    protected function setUp(): void {
        global $DB, $CFG;
        $CFG->enablecompletion = true;

        /** @var \core_badges\testing\generator $generator */
        $generator = $this->getDataGenerator()->get_plugin_generator('core_badges');

        $user = $this->getDataGenerator()->create_user();

        $this->badgeid = $generator->create_badge($user->id, ['status' => BADGE_STATUS_INACTIVE]);

        // Create a course with activity and auto completion tracking.
        $this->course = $this->getDataGenerator()->create_course(array('enablecompletion' => true));
        $this->user = $this->getDataGenerator()->create_user();
        $studentrole = $DB->get_record('role', array('shortname' => 'student'));
        $this->assertNotEmpty($studentrole);

        // Get manual enrolment plugin and enrol user.
        require_once($CFG->dirroot.'/enrol/manual/locallib.php');
        $manplugin = enrol_get_plugin('manual');
        $maninstance = $DB->get_record('enrol', array('courseid' => $this->course->id, 'enrol' => 'manual'), '*', MUST_EXIST);
        $manplugin->enrol_user($maninstance, $this->user->id, $studentrole->id);
        $this->assertEquals(1, $DB->count_records('user_enrolments'));

        $completionsettings = array('completion' => COMPLETION_TRACKING_AUTOMATIC, 'completionview' => 1);
        $this->module = $this->getDataGenerator()->create_module('forum', array('course' => $this->course->id), $completionsettings);

        // Build badge and criteria.
        $this->coursebadge = $generator->create_badge($user->id, [
            'type' => BADGE_TYPE_COURSE,
            'courseid' => $this->course->id,
        ]);

        // Insert Endorsement.
        $endorsement = new stdClass();
        $endorsement->badgeid = $this->coursebadge;
        $endorsement->issuername = "Issuer 123";
        $endorsement->issueremail = "issuer123@email.com";
        $endorsement->issuerurl = "https://example.org/issuer-123";
        $endorsement->dateissued = 1524567747;
        $endorsement->claimid = "https://example.org/robotics-badge.json";
        $endorsement->claimcomment = "Test endorser comment";
        $DB->insert_record('badge_endorsement', $endorsement, true);

        // Insert related badges.
        $badge = new badge($this->coursebadge);
        $clonedid = $badge->make_clone();
        $badgeclone = new badge($clonedid);
        $badgeclone->status = BADGE_STATUS_ACTIVE;
        $badgeclone->save();

        $relatebadge = new stdClass();
        $relatebadge->badgeid = $this->coursebadge;
        $relatebadge->relatedbadgeid = $clonedid;
        $relatebadge->relatedid = $DB->insert_record('badge_related', $relatebadge, true);

        $this->assertion = new stdClass();
        $this->assertion->badge = '{"uid":"%s","recipient":{"identity":"%s","type":"email","hashed":true,"salt":"%s"},"badge":"%s","verify":{"type":"hosted","url":"%s"},"issuedOn":"%d","evidence":"%s"}';
        $this->assertion->class = '{"name":"%s","description":"%s","image":"%s","criteria":"%s","issuer":"%s"}';
        $this->assertion->issuer = '{"name":"%s","url":"%s","email":"%s"}';
        // Format JSON-LD for Openbadge specification version 2.0.
        $this->assertion2 = new stdClass();
        $this->assertion2->badge = '{"recipient":{"identity":"%s","type":"email","hashed":true,"salt":"%s"},' .
            '"badge":{"name":"%s","description":"%s","image":"%s",' .
            '"criteria":{"id":"%s","narrative":"%s"},"issuer":{"name":"%s","url":"%s","email":"%s",' .
            '"@context":"https:\/\/w3id.org\/openbadges\/v2","id":"%s","type":"Issuer"},' .
            '"@context":"https:\/\/w3id.org\/openbadges\/v2","id":"%s","type":"BadgeClass","version":"%s",' .
            '"@language":"en","related":[{"id":"%s","version":"%s","@language":"%s"}],"endorsement":"%s"},' .
            '"verify":{"type":"hosted","url":"%s"},"issuedOn":"%s","evidence":"%s",' .
            '"@context":"https:\/\/w3id.org\/openbadges\/v2","type":"Assertion","id":"%s"}';
        $this->assertion2->class = '{"name":"%s","description":"%s","image":"%s",' .
            '"criteria":{"id":"%s","narrative":"%s"},"issuer":{"name":"%s","url":"%s","email":"%s",' .
            '"@context":"https:\/\/w3id.org\/openbadges\/v2","id":"%s","type":"Issuer"},' .
            '"@context":"https:\/\/w3id.org\/openbadges\/v2","id":"%s","type":"BadgeClass","version":"%s",' .
            '"@language":"%s","related":[{"id":"%s","version":"%s","@language":"%s"}],"endorsement":"%s"}';
        $this->assertion2->issuer = '{"name":"%s","url":"%s","email":"%s",' .
            '"@context":"https:\/\/w3id.org\/openbadges\/v2","id":"%s","type":"Issuer"}';
    }

    public function test_create_badge() {
        $badge = new badge($this->badgeid);

        $this->assertInstanceOf('badge', $badge);
        $this->assertEquals($this->badgeid, $badge->id);
    }

    public function test_clone_badge() {
        $badge = new badge($this->badgeid);
        $newid = $badge->make_clone();
        $clonedbadge = new badge($newid);

        $this->assertEquals($badge->description, $clonedbadge->description);
        $this->assertEquals($badge->issuercontact, $clonedbadge->issuercontact);
        $this->assertEquals($badge->issuername, $clonedbadge->issuername);
        $this->assertEquals($badge->issuercontact, $clonedbadge->issuercontact);
        $this->assertEquals($badge->issuerurl, $clonedbadge->issuerurl);
        $this->assertEquals($badge->expiredate, $clonedbadge->expiredate);
        $this->assertEquals($badge->expireperiod, $clonedbadge->expireperiod);
        $this->assertEquals($badge->type, $clonedbadge->type);
        $this->assertEquals($badge->courseid, $clonedbadge->courseid);
        $this->assertEquals($badge->message, $clonedbadge->message);
        $this->assertEquals($badge->messagesubject, $clonedbadge->messagesubject);
        $this->assertEquals($badge->attachment, $clonedbadge->attachment);
        $this->assertEquals($badge->notification, $clonedbadge->notification);
        $this->assertEquals($badge->version, $clonedbadge->version);
        $this->assertEquals($badge->language, $clonedbadge->language);
        $this->assertEquals($badge->imagecaption, $clonedbadge->imagecaption);
        $this->assertEquals($badge->imageauthorname, $clonedbadge->imageauthorname);
        $this->assertEquals($badge->imageauthoremail, $clonedbadge->imageauthoremail);
        $this->assertEquals($badge->imageauthorurl, $clonedbadge->imageauthorurl);
    }

    public function test_badge_status() {
        $badge = new badge($this->badgeid);
        $old_status = $badge->status;
        $badge->set_status(BADGE_STATUS_ACTIVE);
        $this->assertNotEquals($old_status, $badge->status);
        $this->assertEquals(BADGE_STATUS_ACTIVE, $badge->status);
    }

    public function test_delete_badge() {
        $badge = new badge($this->badgeid);
        $badge->delete();
        // We don't actually delete badges. We archive them.
        $this->assertEquals(BADGE_STATUS_ARCHIVED, $badge->status);
    }

    /**
     * Really delete the badge.
     */
    public function test_delete_badge_for_real() {
        global $DB;

        $badge = new badge($this->badgeid);

        $newid1 = $badge->make_clone();
        $newid2 = $badge->make_clone();
        $newid3 = $badge->make_clone();

        // Insert related badges to badge 1.
        $badge->add_related_badges([$newid1, $newid2, $newid3]);

        // Another badge.
        $badge2 = new badge($newid2);
        // Make badge 1 related for badge 2.
        $badge2->add_related_badges([$this->badgeid]);

        // Confirm that the records about this badge about its relations have been removed as well.
        $relatedsql = 'badgeid = :badgeid OR relatedbadgeid = :relatedbadgeid';
        $relatedparams = array(
            'badgeid' => $this->badgeid,
            'relatedbadgeid' => $this->badgeid
        );
        // Badge 1 has 4 related records. 3 where it's the badgeid, 1 where it's the relatedbadgeid.
        $this->assertEquals(4, $DB->count_records_select('badge_related', $relatedsql, $relatedparams));

        // Delete the badge for real.
        $badge->delete(false);

        // Confirm that the badge itself has been removed.
        $this->assertFalse($DB->record_exists('badge', ['id' => $this->badgeid]));

        // Confirm that the records about this badge about its relations have been removed as well.
        $this->assertFalse($DB->record_exists_select('badge_related', $relatedsql, $relatedparams));
    }

    public function test_create_badge_criteria() {
        $badge = new badge($this->badgeid);
        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL));

        $this->assertCount(1, $badge->get_criteria());

        $criteria_profile = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_PROFILE, 'badgeid' => $badge->id));
        $params = array('agg' => BADGE_CRITERIA_AGGREGATION_ALL, 'field_address' => 'address');
        $criteria_profile->save($params);

        $this->assertCount(2, $badge->get_criteria());
    }

    public function test_add_badge_criteria_description() {
        $criteriaoverall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $this->badgeid));
        $criteriaoverall->save(array(
                'agg' => BADGE_CRITERIA_AGGREGATION_ALL,
                'description' => 'Overall description',
                'descriptionformat' => FORMAT_HTML
        ));
        $criteriaprofile = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_PROFILE, 'badgeid' => $this->badgeid));
        $params = array(
                'agg' => BADGE_CRITERIA_AGGREGATION_ALL,
                'field_address' => 'address',
                'description' => 'Description',
                'descriptionformat' => FORMAT_HTML
        );
        $criteriaprofile->save($params);
        $badge = new badge($this->badgeid);
        $this->assertEquals('Overall description', $badge->criteria[BADGE_CRITERIA_TYPE_OVERALL]->description);
        $this->assertEquals('Description', $badge->criteria[BADGE_CRITERIA_TYPE_PROFILE]->description);
    }

    public function test_delete_badge_criteria() {
        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $this->badgeid));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL));
        $badge = new badge($this->badgeid);

        $this->assertInstanceOf('award_criteria_overall', $badge->criteria[BADGE_CRITERIA_TYPE_OVERALL]);

        $badge->criteria[BADGE_CRITERIA_TYPE_OVERALL]->delete();
        $this->assertEmpty($badge->get_criteria());
    }

    public function test_badge_awards() {
        $badge = new badge($this->badgeid);
        $user1 = $this->getDataGenerator()->create_user();

        $badge->issue($user1->id, true);
        $this->assertTrue($badge->is_issued($user1->id));

        $user2 = $this->getDataGenerator()->create_user();
        $badge->issue($user2->id, true);
        $this->assertTrue($badge->is_issued($user2->id));

        $this->assertCount(2, $badge->get_awards());
    }

    /**
     * Test the {@link badges_get_user_badges()} function in lib/badgeslib.php
     */
    public function test_badges_get_user_badges() {
        global $DB;

        $badges = array();
        $user1 = $this->getDataGenerator()->create_user();
        $user2 = $this->getDataGenerator()->create_user();

        // Record the current time, we need to be precise about a couple of things.
        $now = time();
        // Create 11 badges with which to test.
        for ($i = 1; $i <= 11; $i++) {
            // Mock up a badge.
            $badge = new stdClass();
            $badge->id = null;
            $badge->name = "Test badge $i";
            $badge->description = "Testing badges $i";
            $badge->timecreated = $now - 12;
            $badge->timemodified = $now - 12;
            $badge->usercreated = $user1->id;
            $badge->usermodified = $user1->id;
            $badge->issuername = "Test issuer";
            $badge->issuerurl = "http://issuer-url.domain.co.nz";
            $badge->issuercontact = "issuer@example.com";
            $badge->expiredate = null;
            $badge->expireperiod = null;
            $badge->type = BADGE_TYPE_SITE;
            $badge->courseid = null;
            $badge->messagesubject = "Test message subject for badge $i";
            $badge->message = "Test message body for badge $i";
            $badge->attachment = 1;
            $badge->notification = 0;
            $badge->status = BADGE_STATUS_INACTIVE;
            $badge->version = "Version $i";
            $badge->language = "en";
            $badge->imagecaption = "Image caption $i";
            $badge->imageauthorname = "Image author's name $i";
            $badge->imageauthoremail = "author$i@example.com";
            $badge->imageauthorname = "Image author's name $i";

            $badgeid = $DB->insert_record('badge', $badge, true);
            $badges[$badgeid] = new badge($badgeid);
            $badges[$badgeid]->issue($user2->id, true);
            // Check it all actually worked.
            $this->assertCount(1, $badges[$badgeid]->get_awards());

            // Hack the database to adjust the time each badge was issued.
            // The alternative to this is sleep which is a no-no in unit tests.
            $DB->set_field('badge_issued', 'dateissued', $now - 11 + $i, array('userid' => $user2->id, 'badgeid' => $badgeid));
        }

        // Make sure the first user has no badges.
        $result = badges_get_user_badges($user1->id);
        $this->assertIsArray($result);
        $this->assertCount(0, $result);

        // Check that the second user has the expected 11 badges.
        $result = badges_get_user_badges($user2->id);
        $this->assertCount(11, $result);

        // Test pagination.
        // Ordering is by time issued desc, so things will come out with the last awarded badge first.
        $result = badges_get_user_badges($user2->id, 0, 0, 4);
        $this->assertCount(4, $result);
        $lastbadgeissued = reset($result);
        $this->assertSame('Test badge 11', $lastbadgeissued->name);
        // Page 2. Expecting 4 results again.
        $result = badges_get_user_badges($user2->id, 0, 1, 4);
        $this->assertCount(4, $result);
        $lastbadgeissued = reset($result);
        $this->assertSame('Test badge 7', $lastbadgeissued->name);
        // Page 3. Expecting just three results here.
        $result = badges_get_user_badges($user2->id, 0, 2, 4);
        $this->assertCount(3, $result);
        $lastbadgeissued = reset($result);
        $this->assertSame('Test badge 3', $lastbadgeissued->name);
        // Page 4.... there is no page 4.
        $result = badges_get_user_badges($user2->id, 0, 3, 4);
        $this->assertCount(0, $result);

        // Test search.
        $result = badges_get_user_badges($user2->id, 0, 0, 0, 'badge 1');
        $this->assertCount(3, $result);
        $lastbadgeissued = reset($result);
        $this->assertSame('Test badge 11', $lastbadgeissued->name);
        // The term Totara doesn't appear anywhere in the badges.
        $result = badges_get_user_badges($user2->id, 0, 0, 0, 'Totara');
        $this->assertCount(0, $result);

        // Issue a user with a course badge and verify its returned based on if
        // coursebadges are enabled or disabled.
        $sitebadgeid = key($badges);
        $badges[$sitebadgeid]->issue($this->user->id, true);

        $badge = new stdClass();
        $badge->id = null;
        $badge->name = "Test course badge";
        $badge->description = "Testing course badge";
        $badge->timecreated = $now;
        $badge->timemodified = $now;
        $badge->usercreated = $user1->id;
        $badge->usermodified = $user1->id;
        $badge->issuername = "Test issuer";
        $badge->issuerurl = "http://issuer-url.domain.co.nz";
        $badge->issuercontact = "issuer@example.com";
        $badge->expiredate = null;
        $badge->expireperiod = null;
        $badge->type = BADGE_TYPE_COURSE;
        $badge->courseid = $this->course->id;
        $badge->messagesubject = "Test message subject for course badge";
        $badge->message = "Test message body for course badge";
        $badge->attachment = 1;
        $badge->notification = 0;
        $badge->status = BADGE_STATUS_ACTIVE;
        $badge->version = "Version $i";
        $badge->language = "en";
        $badge->imagecaption = "Image caption";
        $badge->imageauthorname = "Image author's name";
        $badge->imageauthoremail = "author@example.com";
        $badge->imageauthorname = "Image author's name";

        $badgeid = $DB->insert_record('badge', $badge, true);
        $badges[$badgeid] = new badge($badgeid);
        $badges[$badgeid]->issue($this->user->id, true);

        // With coursebadges off, we should only get the site badge.
        set_config('badges_allowcoursebadges', false);
        $result = badges_get_user_badges($this->user->id);
        $this->assertCount(1, $result);

        // With it on, we should get both.
        set_config('badges_allowcoursebadges', true);
        $result = badges_get_user_badges($this->user->id);
        $this->assertCount(2, $result);

    }

    public static function data_for_message_from_template() {
        return array(
            array(
                'This is a message with no variables',
                array(), // no params
                'This is a message with no variables'
            ),
            array(
                'This is a message with %amissing% variables',
                array(), // no params
                'This is a message with %amissing% variables'
            ),
            array(
                'This is a message with %one% variable',
                array('one' => 'a single'),
                'This is a message with a single variable'
            ),
            array(
                'This is a message with %one% %two% %three% variables',
                array('one' => 'more', 'two' => 'than', 'three' => 'one'),
                'This is a message with more than one variables'
            ),
            array(
                'This is a message with %three% %two% %one%',
                array('one' => 'variables', 'two' => 'ordered', 'three' => 'randomly'),
                'This is a message with randomly ordered variables'
            ),
            array(
                'This is a message with %repeated% %one% %repeated% of variables',
                array('one' => 'and', 'repeated' => 'lots'),
                'This is a message with lots and lots of variables'
            ),
        );
    }

    /**
     * @dataProvider data_for_message_from_template
     */
    public function test_badge_message_from_template($message, $params, $result) {
        $this->assertEquals(badge_message_from_template($message, $params), $result);
    }

    /**
     * Assert that the courset generator can handle a badge with > 61 courses
     * @return void
     */
    public function test_badge_courseset_criteria_with_a_huge_number_of_courses(): void {
        // Create more than 61 courses to trigger a mysql DB error
        $generator = $this->getDataGenerator();
        /** @var \core_completion\testing\generator $completion_generator */
        $completion_generator = $this->getDataGenerator()->get_plugin_generator('core_completion');

        // Create the badge
        /** @var \core_badges\testing\generator $badge_generator */
        $badge_generator = $this->getDataGenerator()->get_plugin_generator('core_badges');
        $badge_id = $badge_generator->create_badge(2, []);
        $badge = new badge($badge_id);

        $user = $generator->create_user();
        $user_in = $generator->create_user();
        $user_not = $generator->create_user();
        $user_not_2 = $generator->create_user();

        $courses = [];
        for ($i = 1; $i <= 65; $i++) {
            $course = $generator->create_course();
            $courses[$i] = $course;

            $generator->enrol_user($user->id, $course->id);
            $generator->enrol_user($user_in->id, $course->id);
            $generator->enrol_user($user_not->id, $course->id);
            $generator->enrol_user($user_not_2->id, $course->id);

            // Course 1 will not complete (yet), rest will
            if ($i > 1) {
                $completion_generator->enable_completion_tracking($course);
                $completion_generator->complete_course($course, $user);
                $completion_generator->complete_course($course, $user_in);
                $completion_generator->complete_course($course, $user_not);
            }
        }

        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge_id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ANY));
        $criteria_course = award_criteria::build(['criteriatype' => BADGE_CRITERIA_TYPE_COURSESET, 'badgeid' => $badge_id]);

        $courses_criteria = ['agg' => BADGE_CRITERIA_AGGREGATION_ALL];
        foreach ($courses as $course) {
            $courses_criteria['course_' . $course->id] = $course->id;
        }
        $criteria_course->save($courses_criteria);

        // Run the scheduled task to issue the badge. But the badge should not be issued.
        ob_start();
        $task = manager::get_scheduled_task('core\task\badges_cron_task');
        $task->execute();
        ob_end_clean();

        $this->assertFalse($badge->is_issued($user->id));
        $this->assertFalse($badge->is_issued($user_in->id));
        $this->assertFalse($badge->is_issued($user_not->id));
        $this->assertFalse($badge->is_issued($user_not_2->id));

        // Now complete the missing course for $user alone
        $completion_generator->enable_completion_tracking($courses[1]);
        $completion_generator->complete_course($courses[1], $user);
        $completion_generator->complete_course($courses[1], $user_in);

        // Run the scheduled task to issue the badge. Now the badge should be issued.
        ob_start();
        $task = manager::get_scheduled_task('core\task\badges_cron_task');
        $task->execute();
        ob_end_clean();

        // Check if badge is awarded to $user & $user_in but not $user_not and $user_not_2
        $this->assertTrue($badge->is_issued($user->id));
        $this->assertTrue($badge->is_issued($user_in->id));
        $this->assertFalse($badge->is_issued($user_not->id));
        $this->assertFalse($badge->is_issued($user_not_2->id));
    }

    /**
     * Test for working around the 61 tables join limit of mysql in award_criteria_activity in combination with the scheduled task.
     * @return void
     * @covers \core_badges\badge->review_all_criteria()
     */
    public function test_badge_activity_criteria_with_a_huge_number_of_coursemodules() {
        global $CFG;
        require_once($CFG->dirroot . '/completion/criteria/completion_criteria_activity.php');

        // Create more than 61 modules to potentially trigger an mysql db error.
        $assigncount = 75;
        $assigns = [];
        for ($i = 1; $i <= $assigncount; $i++) {
            $assigns[] = $this->getDataGenerator()->create_module('assign', ['course' => $this->course->id], ['completion' => 1]);
        }
        $assigncmids = array_flip(array_map(function ($assign) {
            return $assign->cmid;
        }, $assigns));
        $criteriaactivityarray = array_fill_keys(array_keys($assigncmids), 1);

        // Set completion criteria.
        $criteriadata = (object) [
            'id' => $this->course->id,
            'criteria_activity' => $criteriaactivityarray,
        ];
        $criterion = new completion_criteria_activity();
        $criterion->update_config($criteriadata);

        $badge = new badge($this->coursebadge);

        $criteriaoverall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteriaoverall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ANY));
        $criteriaactivity = award_criteria::build(['criteriatype' => BADGE_CRITERIA_TYPE_ACTIVITY, 'badgeid' => $badge->id]);

        $modulescrit = ['agg' => BADGE_CRITERIA_AGGREGATION_ALL];
        foreach ($assigns as $assign) {
            $modulescrit['module_' . $assign->cmid] = $assign->cmid;
        }
        $criteriaactivity->save($modulescrit);

        // Take one assign to complete it later.
        $assingtemp = array_shift($assigns);

        // Mark the user to complete the modules.
        foreach ($assigns as $assign) {
            $cmassign = get_coursemodule_from_id('assign', $assign->cmid);
            $completion = new \completion_info($this->course);
            $completion->update_state($cmassign, COMPLETION_COMPLETE, $this->user->id);
        }

        // Run the scheduled task to issue the badge. But the badge should not be issued.
        ob_start();
        $task = manager::get_scheduled_task('core\task\badges_cron_task');
        $task->execute();
        ob_end_clean();

        $this->assertFalse($badge->is_issued($this->user->id));

        // Now complete the last uncompleted module.
        $cmassign = get_coursemodule_from_id('assign', $assingtemp->cmid);
        $completion = new \completion_info($this->course);
        $completion->update_state($cmassign, COMPLETION_COMPLETE, $this->user->id);

        // Run the scheduled task to issue the badge. Now the badge schould be issued.
        ob_start();
        $task = manager::get_scheduled_task('core\task\badges_cron_task');
        $task->execute();
        ob_end_clean();

        // Check if badge is awarded.
        $this->assertTrue($badge->is_issued($this->user->id));
    }

    /**
     * Test badges observer when course module completion event id fired.
     */
    public function test_badges_observer_course_module_criteria_review() {
        $badge = new badge($this->coursebadge);
        $this->assertFalse($badge->is_issued($this->user->id));

        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ANY));
        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_ACTIVITY, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ANY, 'module_'.$this->module->cmid => $this->module->cmid));

        // Assert the badge will not be issued to the user as is.
        $badge = new badge($this->coursebadge);
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));

        // Mark the user as complete by viewing the forum.
        $completioninfo = new completion_info($this->course);
        $coursemodules = get_coursemodules_in_course('forum', $this->course->id);

        $sink = $this->redirectMessages();
        foreach ($coursemodules as $cm) {
            $completioninfo->set_module_viewed($cm, $this->user->id);
        }
        $this->assertCount(1, $sink->get_messages());

        $sink->close();

        // Check if badge is awarded.
        $this->assertTrue($badge->is_issued($this->user->id));
    }

    /**
     * Test badges observer when course_completed event is fired.
     */
    public function test_badges_observer_course_criteria_review() {
        $badge = new badge($this->coursebadge);
        $this->assertFalse($badge->is_issued($this->user->id));

        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ANY));
        $criteria_overall1 = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_COURSE, 'badgeid' => $badge->id));
        $criteria_overall1->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ANY, 'course_'.$this->course->id => $this->course->id));

        $ccompletion = new completion_completion(array('course' => $this->course->id, 'userid' => $this->user->id));

        // Assert the badge will not be issued to the user as is.
        $badge = new badge($this->coursebadge);
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));

        // Mark course as complete.
        $sink = $this->redirectMessages();
        $ccompletion->mark_complete();
        $this->assertCount(1, $sink->get_messages());
        $sink->close();

        // Check if badge is awarded.
        $this->assertTrue($badge->is_issued($this->user->id));
    }

    /**
     * Test badges observer when user_updated event is fired.
     */
    public function test_badges_observer_profile_criteria_review() {
        global $CFG, $DB;
        require_once($CFG->dirroot.'/user/profile/lib.php');

        // Add a custom field of textarea type.
        $customprofileid = $DB->insert_record('user_info_field', array(
            'shortname' => 'newfield', 'name' => 'Description of new field', 'categoryid' => 1,
            'datatype' => 'textarea'));

        $badge = new badge($this->coursebadge);

        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ANY));
        $criteria_overall1 = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_PROFILE, 'badgeid' => $badge->id));
        $criteria_overall1->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL, 'field_address' => 'address', 'field_skype' => 'skype',
            'field_' . $customprofileid => $customprofileid));

        // Assert the badge will not be issued to the user as is.
        $badge = new badge($this->coursebadge);
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));

        // Set the required fields and make sure the badge got issued.
        $this->user->address = 'Test address';
        $this->user->skype = 'user999999999';
        $sink = $this->redirectMessages();
        profile_save_data((object)array('id' => $this->user->id, 'profile_field_newfield' => 'X'));
        user_update_user($this->user, false);
        $this->assertCount(1, $sink->get_messages());
        $sink->close();
        // Check if badge is awarded.
        $this->assertTrue($badge->is_issued($this->user->id));
    }


    /**
     * Test program observer for a site badge when the criteria is for the user to complete all programs (and has).
     */
    public function test_program_observer_for_site_badges_when_user_completes_all_programs_pass() {
        global $CFG, $DB;

        require_once($CFG->dirroot . '/totara/program/lib.php');

        $badge = new badge($this->badgeid);

        // Assert the badge hasn't been issued.
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));

        // Create a program, assign a user to it and mark it as complete.
        $data_generator = $this->getDataGenerator();
        /** @var \totara_program\testing\generator $program_generator */
        $program_generator = $data_generator->get_plugin_generator('totara_program');
        $program1 = $program_generator->create_program();
        $program_generator->assign_to_program($program1->id, assignments::ASSIGNTYPE_INDIVIDUAL, $this->user->id);

        // Prepare the program completion.
        $prog_completion = new stdClass();
        $prog_completion->programid = $program1->id;
        $prog_completion->userid = $this->user->id;
        $prog_completion->status = program::STATUS_PROGRAM_COMPLETE;
        $prog_completion->timedue = 1003;
        $prog_completion->timecompleted = 1001;
        $prog_completion->organisationid = 0;
        $prog_completion->positionid = 0;

        // Update the program completion and check that was successful.
        $prog_completion->id = $DB->get_field('prog_completion', 'id',
            array('programid' => $program1->id, 'userid' => $this->user->id, 'coursesetid' => 0));
        $result = prog_write_completion($prog_completion);
        $this->assertTrue($result);
        $program_completions = $DB->record_exists('prog_completion',
            array('programid' => $program1->id, 'userid' => $this->user->id, 'coursesetid' => 0, 'status' => program::STATUS_PROGRAM_COMPLETE));
        $this->assertTrue($program_completions);

        // Create a second program.
        $program2 = $program_generator->create_program();
        $program_generator->assign_to_program($program2->id, assignments::ASSIGNTYPE_INDIVIDUAL, $this->user->id, null, true);

        // Prepare the second program completion.
        $prog_completion2 = clone($prog_completion);
        $prog_completion2->programid = $program2->id;
        $prog_completion2->userid = $this->user->id;

        // Update the program completion and check that was successful.
        $prog_completion2->id = $DB->get_field('prog_completion', 'id',
            array('programid' => $program2->id, 'userid' => $this->user->id, 'coursesetid' => 0));
        $result = prog_write_completion($prog_completion2);
        $this->assertTrue($result);
        $program_completions = $DB->record_exists('prog_completion',
            array('programid' => $program2->id, 'userid' => $this->user->id, 'coursesetid' => 0, 'status' => program::STATUS_PROGRAM_COMPLETE));
        $this->assertTrue($program_completions);

        // Save the criteria for the badge.
        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL));
        $criteria_overall1 = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_PROGRAM, 'badgeid' => $badge->id));
        $criteria_overall1->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL, 'program_programs' => array($program1->id, $program2->id)));

        // Assert the badge has been issued.
        $badge = new badge($this->badgeid);
        $badge->review_all_criteria();
        $this->assertTrue($badge->is_issued($this->user->id));
    }

    /**
     * Test program observer for a site badge when the criteria is for the user to complete all programs but hasn't.
     */
    public function test_program_observer_for_site_badges_when_user_completes_all_programs_fail() {
        global $CFG, $DB;

        require_once($CFG->dirroot . '/totara/program/lib.php');

        $badge = new badge($this->badgeid);

        // Assert the badge hasn't been issued.
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));

        // Create a program, assign a user to it and mark it as complete.
        $data_generator = $this->getDataGenerator();
        /** @var \totara_program\testing\generator $program_generator */
        $program_generator = $data_generator->get_plugin_generator('totara_program');
        $program1 = $program_generator->create_program();
        $program_generator->assign_to_program($program1->id, assignments::ASSIGNTYPE_INDIVIDUAL, $this->user->id);

        // Prepare the program completion.
        $prog_completion = new stdClass();
        $prog_completion->programid = $program1->id;
        $prog_completion->userid = $this->user->id;
        $prog_completion->status = program::STATUS_PROGRAM_COMPLETE;
        $prog_completion->timedue = 1003;
        $prog_completion->timecompleted = 1001;
        $prog_completion->organisationid = 0;
        $prog_completion->positionid = 0;

        // Update the program completion and check that was successful.
        $prog_completion->id = $DB->get_field('prog_completion', 'id',
            array('programid' => $program1->id, 'userid' => $this->user->id, 'coursesetid' => 0));
        $result = prog_write_completion($prog_completion);
        $this->assertTrue($result);
        $program_completions = $DB->record_exists('prog_completion',
            array('programid' => $program1->id, 'userid' => $this->user->id, 'coursesetid' => 0, 'status' => program::STATUS_PROGRAM_COMPLETE));
        $this->assertTrue($program_completions);

        // Create a second user.
        $user = $this->getDataGenerator()->create_user();

        // Create a second program.
        $program2 = $program_generator->create_program();
        $program_generator->assign_to_program($program2->id, assignments::ASSIGNTYPE_INDIVIDUAL, $user->id);

        // Prepare the second program completion.
        $prog_completion2 = clone($prog_completion);
        $prog_completion2->programid = $program2->id;
        $prog_completion2->userid = $user->id;

        // Update the program completion and check that was successful.
        $prog_completion2->id = $DB->get_field('prog_completion', 'id',
            array('programid' => $program2->id, 'userid' => $user->id, 'coursesetid' => 0));
        $result = prog_write_completion($prog_completion2);
        $this->assertTrue($result);
        $program_completions = $DB->record_exists('prog_completion',
            array('programid' => $program2->id, 'userid' => $user->id, 'coursesetid' => 0, 'status' => program::STATUS_PROGRAM_COMPLETE));
        $this->assertTrue($program_completions);

        // Save the criteria for the badge.
        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL));
        $criteria_overall1 = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_PROGRAM, 'badgeid' => $badge->id));
        $criteria_overall1->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL, 'program_programs' => array($program1->id, $program2->id)));

        // Assert the badge has been issued.
        $badge = new badge($this->badgeid);
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));
    }


    /**
     * Test program observer for a site badge when the criteria is for the user to complete any programs (and has).
     */
    public function test_program_observer_for_site_badges_when_user_completes_any_programs_pass() {
        global $CFG, $DB;

        require_once($CFG->dirroot . '/totara/program/lib.php');

        $badge = new badge($this->badgeid);

        // Assert the badge hasn't been issued.
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));

        // Create a program, assign a user to it and mark it as complete.
        $data_generator = $this->getDataGenerator();
        /** @var \totara_program\testing\generator $program_generator */
        $program_generator = $data_generator->get_plugin_generator('totara_program');
        $program1 = $program_generator->create_program();
        $program_generator->assign_to_program($program1->id, assignments::ASSIGNTYPE_INDIVIDUAL, $this->user->id);

        // Prepare the program completion.
        $prog_completion = new stdClass();
        $prog_completion->programid = $program1->id;
        $prog_completion->userid = $this->user->id;
        $prog_completion->status = program::STATUS_PROGRAM_COMPLETE;
        $prog_completion->timedue = 1003;
        $prog_completion->timecompleted = 1001;
        $prog_completion->organisationid = 0;
        $prog_completion->positionid = 0;

        // Update the program completion and check that was successful.
        $prog_completion->id = $DB->get_field('prog_completion', 'id',
            array('programid' => $program1->id, 'userid' => $this->user->id, 'coursesetid' => 0));
        $result = prog_write_completion($prog_completion);
        $this->assertTrue($result);
        $program_completions = $DB->record_exists('prog_completion',
            array('programid' => $program1->id, 'userid' => $this->user->id, 'coursesetid' => 0, 'status' => program::STATUS_PROGRAM_COMPLETE));
        $this->assertTrue($program_completions);

        // Create a second program. Don't create a completion for this one.
        $program2 = $program_generator->create_program();
        $program_generator->assign_to_program($program2->id, assignments::ASSIGNTYPE_INDIVIDUAL, $this->user->id);

        // Save the criteria for the badge.
        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL));
        $criteria_overall1 = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_PROGRAM, 'badgeid' => $badge->id));
        $criteria_overall1->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ANY, 'program_programs' => array($program1->id, $program2->id)));

        // Assert the badge has been issued.
        $badge = new badge($this->badgeid);
        $badge->review_all_criteria();
        $this->assertTrue($badge->is_issued($this->user->id));
    }

    /**
     * Test program observer for a site badge when the criteria is for the user to complete all programs but hasn't.
     */
    public function test_program_observer_for_site_badges_when_user_completes_any_programs_fail() {
        global $CFG, $DB;

        require_once($CFG->dirroot . '/totara/program/lib.php');

        $badge = new badge($this->badgeid);

        // Assert the badge hasn't been issued.
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));

        // Create a program, assign a user to it. Don't add a completion for either programs.
        $data_generator = $this->getDataGenerator();
        /** @var \totara_program\testing\generator $program_generator */
        $program_generator = $data_generator->get_plugin_generator('totara_program');
        $program1 = $program_generator->create_program();
        $program_generator->assign_to_program($program1->id, assignments::ASSIGNTYPE_INDIVIDUAL, $this->user->id);

        // Create a second program.
        $program2 = $program_generator->create_program();
        $program_generator->assign_to_program($program1->id, assignments::ASSIGNTYPE_INDIVIDUAL, $this->user->id);

        // Save the criteria for the badge.
        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL));
        $criteria_overall1 = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_PROGRAM, 'badgeid' => $badge->id));
        $criteria_overall1->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ANY, 'program_programs' => array($program1->id, $program2->id)));

        // Assert the badge has been issued.
        $badge = new badge($this->badgeid);
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));
    }


    /**
     * Test cohort observer for a site badge when the criteria is for the user to be a member of all cohorts (and is).
     */
    public function test_cohort_observer_for_site_badges_with_user_in_all_cohorts_pass() {
        global $DB;

        $badge = new badge($this->badgeid);

        // Assert the badge hasn't been issued.
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));

        // Create a cohort and add the user to it.
        $cohort1 = $this->getDataGenerator()->create_cohort();
        cohort_add_member($cohort1->id, $this->user->id);
        // Check that the user exists on the cohort.
        $this->assertTrue($DB->record_exists('cohort_members', array('cohortid'=>$cohort1->id, 'userid'=>$this->user->id)));

        // Create a second cohort and add the user to it.
        $cohort2 = $this->getDataGenerator()->create_cohort();
        cohort_add_member($cohort2->id, $this->user->id);
        // Check that the user exists on the cohort.
        $this->assertTrue($DB->record_exists('cohort_members', array('cohortid'=>$cohort2->id, 'userid'=>$this->user->id)));

        // Save the criteria for the badge.
        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL));
        $criteria_overall1 = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_COHORT, 'badgeid' => $badge->id));
        $criteria_overall1->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL, 'cohort_cohorts' => array($cohort1->id, $cohort2->id)));

        // Assert the badge has been issued.
        $badge = new badge($this->badgeid);
        $badge->review_all_criteria();
        $this->assertTrue($badge->is_issued($this->user->id));
    }

    /**
     * Test cohort observer for a site badge when the criteria is for the user to be a member of all cohorts but isn't.
     */
    public function test_cohort_observer_for_site_badges_with_user_in_all_cohorts_fail() {
        global $DB;

        $badge = new badge($this->badgeid);

        // Assert the badge hasn't been issued.
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));

        // Create a cohort and add the user to it.
        $cohort1 = $this->getDataGenerator()->create_cohort();
        cohort_add_member($cohort1->id, $this->user->id);
        // Check that the user exists on the cohort.
        $this->assertTrue($DB->record_exists('cohort_members', array('cohortid'=>$cohort1->id, 'userid'=>$this->user->id)));

        // Create a second cohort and add a second user to it.
        $cohort2 = $this->getDataGenerator()->create_cohort();
        $user = $this->getDataGenerator()->create_user();
        cohort_add_member($cohort2->id, $user->id);
        // Check that the user exists on the cohort.
        $this->assertTrue($DB->record_exists('cohort_members', array('cohortid'=>$cohort2->id, 'userid'=>$user->id)));

        // Save the criteria for the badge.
        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL));
        $criteria_overall1 = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_COHORT, 'badgeid' => $badge->id));
        $criteria_overall1->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL, 'cohort_cohorts' => array($cohort1->id, $cohort2->id)));

        // Assert the badge has not been issued.
        $badge = new badge($this->badgeid);
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));
    }

    /**
     * Test cohort observer for a site badge when the criteria is for the user to be a member of any cohort (and is).
     */
    public function test_cohort_observer_for_site_badges_with_user_in_any_cohort_pass() {
        global $DB;

        $badge = new badge($this->badgeid);

        // Assert the badge hasn't been issued.
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));

        // Create a cohort and add the user to it.
        $cohort1 = $this->getDataGenerator()->create_cohort();
        cohort_add_member($cohort1->id, $this->user->id);
        // Check that the user exists on the cohort.
        $this->assertTrue($DB->record_exists('cohort_members', array('cohortid'=>$cohort1->id, 'userid'=>$this->user->id)));

        // Create a second cohort but don't add the user to it.
        $cohort2 = $this->getDataGenerator()->create_cohort();

        // Save the criteria for the badge.
        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL));
        $criteria_overall1 = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_COHORT, 'badgeid' => $badge->id));
        $criteria_overall1->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ANY, 'cohort_cohorts' => array($cohort1->id, $cohort2->id)));

        // Assert the badge has been issued.
        $badge = new badge($this->badgeid);
        $badge->review_all_criteria();
        $this->assertTrue($badge->is_issued($this->user->id));
    }

    /**
     * Test cohort observer for a site badge when the criteria is for the user to be a member of any cohort but isn't.
     */
    public function test_cohort_observer_for_site_badges_with_user_in_any_cohort_fail() {
        global $DB;

        $badge = new badge($this->badgeid);

        // Assert the badge hasn't been issued.
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));

        // Create a couple of cohorts. Don't add any other users as they'll be awarded badges.
        $cohort1 = $this->getDataGenerator()->create_cohort();
        $cohort2 = $this->getDataGenerator()->create_cohort();

        // Save the criteria for the badge.
        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL));
        $criteria_overall1 = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_COHORT, 'badgeid' => $badge->id));
        $criteria_overall1->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ANY, 'cohort_cohorts' => array($cohort1->id, $cohort2->id)));

        // Assert the badge has not been issued.
        $badge = new badge($this->badgeid);
        $badge->review_all_criteria();
        $this->assertFalse($badge->is_issued($this->user->id));
    }

    /**
     * Test badges assertion generated when a badge is issued.
     */
    public function test_badges_assertion() {
        $badge = new badge($this->coursebadge);
        $this->assertFalse($badge->is_issued($this->user->id));

        $criteria_overall = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_OVERALL, 'badgeid' => $badge->id));
        $criteria_overall->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ANY));
        $criteria_overall1 = award_criteria::build(array('criteriatype' => BADGE_CRITERIA_TYPE_PROFILE, 'badgeid' => $badge->id));
        $criteria_overall1->save(array('agg' => BADGE_CRITERIA_AGGREGATION_ALL, 'field_address' => 'address'));

        $this->user->address = 'Test address';
        $sink = $this->redirectMessages();
        user_update_user($this->user, false);
        $this->assertCount(1, $sink->get_messages());
        $sink->close();
        // Check if badge is awarded.
        $awards = $badge->get_awards();
        $this->assertCount(1, $awards);

        // Get assertion.
        $award = reset($awards);
        $assertion = new core_badges_assertion($award->uniquehash, OPEN_BADGES_V1);
        $testassertion = $this->assertion;

        // Make sure JSON strings have the same structure.
        $this->assertStringMatchesFormat($testassertion->badge, json_encode($assertion->get_badge_assertion()));
        $this->assertStringMatchesFormat($testassertion->class, json_encode($assertion->get_badge_class()));
        $this->assertStringMatchesFormat($testassertion->issuer, json_encode($assertion->get_issuer()));

        // Test Openbadge specification version 2.
        // Get assertion version 2.
        $award = reset($awards);
        $assertion2 = new core_badges_assertion($award->uniquehash, OPEN_BADGES_V2);
        $testassertion2 = $this->assertion2;

        // Make sure JSON strings have the same structure.
        $this->assertStringMatchesFormat($testassertion2->badge, json_encode($assertion2->get_badge_assertion()));
        $this->assertStringMatchesFormat($testassertion2->class, json_encode($assertion2->get_badge_class()));
        $this->assertStringMatchesFormat($testassertion2->issuer, json_encode($assertion2->get_issuer()));
    }

    /**
     * Tests the core_badges_myprofile_navigation() function.
     */
    public function test_core_badges_myprofile_navigation() {
        // Set up the test.
        $tree = new \core_user\output\myprofile\tree();
        $this->setAdminUser();
        $badge = new badge($this->badgeid);
        $badge->issue($this->user->id, true);
        $iscurrentuser = true;
        $course = null;

        // Enable badges.
        set_config('enablebadges', true);

        // Check the node tree is correct.
        core_badges_myprofile_navigation($tree, $this->user, $iscurrentuser, $course);
        $reflector = new ReflectionObject($tree);
        $nodes = $reflector->getProperty('nodes');
        $nodes->setAccessible(true);
        $this->assertArrayHasKey('localbadges', $nodes->getValue($tree));
    }

    /**
     * Tests the core_badges_myprofile_navigation() function with badges disabled..
     */
    public function test_core_badges_myprofile_navigation_badges_disabled() {
        // Set up the test.
        $tree = new \core_user\output\myprofile\tree();
        $this->setAdminUser();
        $badge = new badge($this->badgeid);
        $badge->issue($this->user->id, true);
        $iscurrentuser = false;
        $course = null;

        // Disable badges.
        set_config('enablebadges', false);

        // Check the node tree is correct.
        core_badges_myprofile_navigation($tree, $this->user, $iscurrentuser, $course);
        $reflector = new ReflectionObject($tree);
        $nodes = $reflector->getProperty('nodes');
        $nodes->setAccessible(true);
        $this->assertArrayNotHasKey('localbadges', $nodes->getValue($tree));
    }

    /**
     * Tests the core_badges_myprofile_navigation() function with a course badge.
     */
    public function test_core_badges_myprofile_navigation_with_course_badge() {
        // Set up the test.
        $tree = new \core_user\output\myprofile\tree();
        $this->setAdminUser();
        $badge = new badge($this->coursebadge);
        $badge->issue($this->user->id, true);
        $iscurrentuser = false;

        // Check the node tree is correct.
        core_badges_myprofile_navigation($tree, $this->user, $iscurrentuser, $this->course);
        $reflector = new ReflectionObject($tree);
        $nodes = $reflector->getProperty('nodes');
        $nodes->setAccessible(true);
        $this->assertArrayHasKey('localbadges', $nodes->getValue($tree));
    }

    /**
     * Test insert and update endorsement with a site badge.
     */
    public function test_badge_endorsement() {
        $badge = new badge($this->badgeid);

        // Insert Endorsement.
        $endorsement = new stdClass();
        $endorsement->badgeid = $this->badgeid;
        $endorsement->issuername = "Issuer 123";
        $endorsement->issueremail = "issuer123@email.com";
        $endorsement->issuerurl = "https://example.org/issuer-123";
        $endorsement->dateissued = 1524567747;
        $endorsement->claimid = "https://example.org/robotics-badge.json";
        $endorsement->claimcomment = "Test endorser comment";

        $badge->save_endorsement($endorsement);
        $endorsement1 = $badge->get_endorsement();
        $this->assertEquals($endorsement->badgeid, $endorsement1->badgeid);
        $this->assertEquals($endorsement->issuername, $endorsement1->issuername);
        $this->assertEquals($endorsement->issueremail, $endorsement1->issueremail);
        $this->assertEquals($endorsement->issuerurl, $endorsement1->issuerurl);
        $this->assertEquals($endorsement->dateissued, $endorsement1->dateissued);
        $this->assertEquals($endorsement->claimid, $endorsement1->claimid);
        $this->assertEquals($endorsement->claimcomment, $endorsement1->claimcomment);

        // Update Endorsement.
        $endorsement1->issuername = "Issuer update";
        $badge->save_endorsement($endorsement1);
        $endorsement2 = $badge->get_endorsement();
        $this->assertEquals($endorsement1->id, $endorsement2->id);
        $this->assertEquals($endorsement1->issuername, $endorsement2->issuername);
    }

    /**
     * Test insert and delete related badge with a site badge.
     */
    public function test_badge_related() {
        $badge = new badge($this->badgeid);
        $newid1 = $badge->make_clone();
        $newid2 = $badge->make_clone();
        $newid3 = $badge->make_clone();

        // Insert an related badge.
        $badge->add_related_badges([$newid1, $newid2, $newid3]);
        $this->assertCount(3, $badge->get_related_badges());

        // Only get related is active.
        $clonedbage1 = new badge($newid1);
        $clonedbage1->status = BADGE_STATUS_ACTIVE;
        $clonedbage1->save();
        $this->assertCount(1, $badge->get_related_badges(true));

        // Delete an related badge.
        $badge->delete_related_badge($newid2);
        $this->assertCount(2, $badge->get_related_badges());
    }

    /**
     * Make sure criteria validation for programs doesn't fail for more than 10 programs (this used to be a bug).
     */
    public function test_program_criteria_validation_for_more_than_10_programs(): void {
        // Create 11 programs.
        /** @var \totara_program\testing\generator $program_generator */
        $program_generator = self::getDataGenerator()->get_plugin_generator('totara_program');
        $program_ids = [];
        for ($i = 0; $i < 11; $i ++) {
            $program_ids[] = $program_generator->create_program()->id;
        }

        // Save the criteria for the badge.
        $badge = new badge($this->badgeid);
        award_criteria::build([
            'criteriatype' => BADGE_CRITERIA_TYPE_PROGRAM,
            'badgeid' => $badge->id
        ])->save([
            'agg' => BADGE_CRITERIA_AGGREGATION_ALL,
            'program_programs' => $program_ids
        ]);

        $criteria = $badge->get_criteria();
        $program_criterion = $criteria[BADGE_CRITERIA_TYPE_PROGRAM];
        self::assertCount(11, $program_criterion->params);

        [$valid, $error_message] = $program_criterion->validate();
        self::assertTrue($valid);
        self::assertEmpty($error_message);
    }

    // Check get_site_backpack() - this method returns one record.
    public function test_badges_get_site_backpack_decrypts_password_only_when_necessary(): void {
        // Some records are created on installation in the badge_external_backpack table.
        $record = builder::table('badge_external_backpack')->find(1);
        static::assertNotEmpty($record);
        // password field is empty by default.
        static::assertSame('', $record->password);

        // Password should also be empty string when fetching via badges_get_site_backpack().
        $record = badges_get_site_backpack(1);
        static::assertSame('', $record->password);

        // Update the record with an unencrypted password.
        builder::table('badge_external_backpack')
            ->where('id', 1)
            ->update(['password' => 'clear_text_password']);

        // It should return the clear text password as it is in DB.
        $record = badges_get_site_backpack(1);
        static::assertSame('clear_text_password', $record->password);

        // Update the record with an encrypted password.
        $encrypted_password = encryption_helper::encrypt_string('secret_password', '1', encryption_helper::ITEM_TYPE_EXTERNAL);
        builder::table('badge_external_backpack')
            ->where('id', 1)
            ->update(['password' => $encrypted_password]);

        // Make sure it's not clear text in the DB.
        $record = builder::table('badge_external_backpack')->find(1);
        static::assertNotEmpty($record->password);
        static::assertNotEquals('secret_password', $record->password);

        // It should be decrypted to clear text when using badges_get_site_backpack().
        $record = badges_get_site_backpack(1);
        static::assertSame('secret_password', $record->password);
    }

    // Check get_site_backpacks() - this method returns all the records.
    public function test_badges_get_site_backpacks_decrypts_password_only_when_necessary(): void {
        // There should be more than two records by default in the badge_external_backpack table.
        $records_from_db = builder::table('badge_external_backpack')->get();
        static::assertGreaterThan(2, $records_from_db->count());

        // Fetching via badges_get_site_backpacks() should return the same number of results.
        $records_from_function = badges_get_site_backpacks();
        static::assertCount($records_from_db->count(), $records_from_function);

        // By default, all password fields should be empty string.
        $records_from_db->map(fn (stdClass $record) => static::assertSame('', $record->password));

        // Passwords should also be empty string when fetching via badges_get_site_backpacks().
        foreach ($records_from_function as $record) {
            static::assertSame('', $record->password);
        }

        // Pick two records to manipulate.
        $record1 = $records_from_db->first();
        $record2 = $records_from_db->last();

        // Update one record with an unencrypted password.
        builder::table('badge_external_backpack')
            ->where('id', $record1->id)
            ->update(['password' => 'clear_text_password1']);

        // Update one record with an encrypted password.
        $encrypted_password = encryption_helper::encrypt_string('secret_password', $record2->id, encryption_helper::ITEM_TYPE_EXTERNAL);
        builder::table('badge_external_backpack')
            ->where('id', $record2->id)
            ->update(['password' => $encrypted_password]);

        // Make sure all the records are returned with their expected passwords.
        $records_from_function = badges_get_site_backpacks();
        $found_unencrypted_pw = false;
        $found_encrypted_pw = false;
        foreach ($records_from_function as $record) {
            if ((int)$record->id === (int)$record1->id) {
                $expected_password = 'clear_text_password1';
                $found_unencrypted_pw = true;
            } elseif ((int)$record->id === (int)$record2->id) {
                $expected_password = 'secret_password';
                $found_encrypted_pw = true;
            } else {
                $expected_password = '';
            }
            static::assertSame($expected_password, $record->password);
        }
        static::assertTrue($found_unencrypted_pw);
        static::assertTrue($found_encrypted_pw);
    }

    public function test_badges_update_site_backpack_encrypts_password(): void {
        static::setAdminUser();

        $record = builder::table('badge_external_backpack')->get()->first();

        $update_data = (object)[
            'password' => 'should_be_encrypted',
            'apiversion' => $record->apiversion,
            'backpackweburl' => $record->backpackweburl,
            'backpackapiurl' => $record->backpackapiurl,
            'oauth2_issuerid' => $record->oauth2_issuerid,
        ];
        $result = badges_update_site_backpack($record->id, $update_data);
        static::assertTrue($result);

        $record_updated = builder::table('badge_external_backpack')->find($record->id);
        static::assertNotEmpty($record_updated->password);
        // Make sure it's not saved in clear text.
        static::assertNotEquals('should_be_encrypted', $record_updated->password);
        // Make sure it can be decrypted as expected.
        static::assertEquals(
            'should_be_encrypted',
            encryption_helper::decrypt_string($record_updated->password, $record->id, encryption_helper::ITEM_TYPE_EXTERNAL)
        );

        // Make sure it works together with badges_get_site_backpack().
        /** @var stdClass $record_from_function */
        $record_from_function = badges_get_site_backpack($record->id);
        static::assertEquals('should_be_encrypted', $record_from_function->password);

        // Make sure it works together with badges_get_site_backpacks().
        $all_records = badges_get_site_backpacks();
        static::assertEquals('should_be_encrypted', $all_records[$record->id]->password);
    }

    public function test_badges_update_site_backpack_does_not_double_encrypt(): void {
        static::setAdminUser();

        $record = builder::table('badge_external_backpack')->get()->first();

        $already_encrypted = encryption_helper::encrypt_string('secret_password', $record->id, encryption_helper::ITEM_TYPE_EXTERNAL);

        $update_data = (object)[
            'password' => $already_encrypted,
            'apiversion' => $record->apiversion,
            'backpackweburl' => $record->backpackweburl,
            'backpackapiurl' => $record->backpackapiurl,
            'oauth2_issuerid' => $record->oauth2_issuerid,
        ];
        $result = badges_update_site_backpack($record->id, $update_data);
        static::assertTrue($result);

        $record_updated = builder::table('badge_external_backpack')->find($record->id);
        static::assertNotEmpty($record_updated->password);
        // Make sure it can be decrypted as expected.
        static::assertEquals(
            'secret_password',
            encryption_helper::decrypt_string($record_updated->password, $record->id, encryption_helper::ITEM_TYPE_EXTERNAL)
        );
    }

    public function test_badges_create_backpack_encrypts_password(): void {
        $builder = builder::table('badge_backpack');
        static::assertSame(0, $builder->count());

        // Create two records.
        $data1 = [
            'userid' => 111,
            'email' => 'test1@example.com',
            'backpackuid' => 222,
            'autosync' => 1,
            'password' => 'my_test_password1',
            'externalbackpackid' => 333,
        ];
        $id1 = badges_create_backpack((object)$data1);

        $data2 = [
            'userid' => 444,
            'email' => 'test2@example.com',
            'backpackuid' => 555,
            'autosync' => 0,
            'password' => 'my_test_password2',
            'externalbackpackid' => 666,
        ];
        $id2 = badges_create_backpack((object)$data2);

        static::assertSame(2, $builder->count());

        // Check first record
        $record1 = $builder->find($id1);

        // Make sure password was encrypted
        static::assertNotEmpty($record1->password);
        static::assertNotEquals('my_test_password1', $record1->password);
        static::assertSame('my_test_password1', encryption_helper::decrypt_string($record1->password, $id1, encryption_helper::ITEM_TYPE_USER));

        // Check the rest of the data.
        $expected_record = $data1;
        unset($expected_record['password'], $record1->password);
        $expected_record['id'] = $id1;

        // Sort the arrays by key for comparison
        ksort($expected_record);
        $actual_record = (array)$record1;
        ksort($actual_record);
        static::assertEquals($expected_record, $actual_record);

        // Check second record
        $record2 = $builder->find($id2);

        // Make sure password was encrypted
        static::assertNotEmpty($record2->password);
        static::assertNotEquals('my_test_password2', $record2->password);
        static::assertSame('my_test_password2', encryption_helper::decrypt_string($record2->password, $id2, encryption_helper::ITEM_TYPE_USER));

        // Check the rest of the data.
        $expected_record = $data2;
        unset($expected_record['password'], $record2->password);
        $expected_record['id'] = $id2;

        // Sort the arrays by key for comparison
        ksort($expected_record);
        $actual_record = (array)$record2;
        ksort($actual_record);
        static::assertEquals($expected_record, $actual_record);
    }

    public function test_badges_create_backpack_does_not_encrypt_empty_password(): void {
        $data = [
            'userid' => 111,
            'email' => 'test1@example.com',
            'backpackuid' => 222,
            'autosync' => 1,
            'password' => '',
            'externalbackpackid' => 333,
        ];
        $id = badges_create_backpack((object)$data);

        $record = builder::table('badge_backpack')->find($id);
        static::assertSame('', $record->password);
    }
}
