<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Matthias Bonk <matthias.bonk@totara.com>
 * @package core_badges
 */

use core_badges\entity\badge_email_verify as badge_email_verify_entity;
use core_badges\model\badge_email_verify;
use core_badges\userdata\email_verification;
use core_phpunit\testcase;
use totara_userdata\userdata\item;
use totara_userdata\userdata\target_user;

defined('MOODLE_INTERNAL') || die();

/**
 * Tests the issued badges userdata.
 */
class core_badges_userdata_email_verification_test extends testcase {

    /**
     * Test it's purgeable in all statuses.
     */
    public function test_is_purgeable(): void {
        self::assertTrue(email_verification::is_purgeable(target_user::STATUS_ACTIVE));
        self::assertTrue(email_verification::is_purgeable(target_user::STATUS_DELETED));
        self::assertTrue(email_verification::is_purgeable(target_user::STATUS_SUSPENDED));
    }

    /**
     * Test compatible context levels
     */
    public function test_compatible_context_levels(): void {
        $this->assertEquals([CONTEXT_SYSTEM], email_verification::get_compatible_context_levels());
    }

    /**
     * Test counting.
     */
    public function test_count(): void {
        [$user1, $user2] = $this->create_test_data();

        $target_user1 = new target_user($user1);
        $target_user2 = new target_user($user2);

        self::assertSame(1, email_verification::execute_count($target_user1, context_system::instance()));
        self::assertSame(1, email_verification::execute_count($target_user2, context_system::instance()));

        // Count for a user without a badge_email_verify record.
        $user3 = static::getDataGenerator()->create_user();
        $target_user3 = new target_user($user3);
        self::assertSame(0, email_verification::execute_count($target_user3, context_system::instance()));
    }

    /**
     * Test purging.
     */
    public function test_purge(): void {
        [$user1, $user2] = $this->create_test_data();

        $target_user1 = new target_user($user1);
        $target_user2 = new target_user($user2);

        self::assertSame(item::RESULT_STATUS_SUCCESS, email_verification::execute_purge($target_user1, context_system::instance()));

        self::assertSame(0, email_verification::execute_count($target_user1, context_system::instance()));
        self::assertSame(1, email_verification::execute_count($target_user2, context_system::instance()));
        static::assertSame(0, badge_email_verify_entity::repository()->where('user_id', $target_user1->id)->count());
        static::assertSame(1, badge_email_verify_entity::repository()->where('user_id', $target_user2->id)->count());

        self::assertSame(item::RESULT_STATUS_SUCCESS, email_verification::execute_purge($target_user2, context_system::instance()));

        self::assertSame(0, email_verification::execute_count($target_user1, context_system::instance()));
        self::assertSame(0, email_verification::execute_count($target_user2, context_system::instance()));
        static::assertSame(0, badge_email_verify_entity::repository()->count());
    }

    /**
     * @return array
     */
    private function create_test_data(): array {
        $user1 = static::getDataGenerator()->create_user();
        $user2 = static::getDataGenerator()->create_user();

        static::setUser($user1);
        badge_email_verify::create('test1@example.com', '111', 'password1', 'secret1');
        static::setUser($user2);
        badge_email_verify::create('test2@example.com', '222', 'password2', 'secret2');

        return [$user1, $user2];
    }
}