<?php
/**
 * This file is part of Totara Core
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Cody Finegan <cody.finegan@totara.com>
 * @package core_mfa
 */
defined('MOODLE_INTERNAL') || die();

/**
 * @group totara_reportbuilder
 */
class core_mfa_rb_user_mfa_test extends \core_phpunit\testcase {
    use totara_reportbuilder\phpunit\report_testing;

    /**
     * Confirm that users can be filtered by their MFA state
     *
     * @return void
     */
    public function test_report(): void {
        global $CFG, $DB;

        $user1 = $this->getDataGenerator()->create_user(['username' => 'nomfa']);
        $user2 = $this->getDataGenerator()->create_user(['username' => 'hasmfa']);

        // Enable a mock MFA for user 2
        $mfa = new \stdClass();
        $mfa->user_id = $user2->id;
        $mfa->type = 'button';
        $mfa->config = '{}';
        $mfa->created_at = time();
        $mfa->updated_at = time();
        $DB->insert_record('mfa_instance_config', $mfa);

        $this->setAdminUser();

        $rid = $this->create_report('user', 'User MFA');

        $config = (new rb_config())->set_nocache(true);
        $report = reportbuilder::create($rid, $config);
        $this->add_column($report, 'user', 'username', null, null, null, 0);
        $this->add_column($report, 'user', 'mfaenabled', null, null, null, 0);
        $this->add_filter($report, 'user', 'mfaenabled', 0, 'Has MFA', 0, 0);

        $report = reportbuilder::create($rid, $config);
        [$sql, $params,] = $report->build_query();

        // Both users + admin + guest
        $records = $DB->get_records_sql($sql, $params);
        $this->assertCount(4, $records);

        $user1_record = null;
        $user2_record = null;
        foreach ($records as $record) {
            if ($record->id == $user1->id) {
                $user1_record = $record;
            }
            if ($record->id == $user2->id) {
                $user2_record = $record;
            }
        }
        $this->assertNotNull($user1_record);
        $this->assertNotNull($user2_record);

        $this->assertEquals('0', $user1_record->user_mfaenabled);
        $this->assertEquals('1', $user2_record->user_mfaenabled);

        // Apply the filter
        /** @var rb_filter_correlated_subquery_select $mfa_filter */
        $mfa_filter = null;
        /** @var \rb_filter_option $filter */
        foreach ($report->get_filters() as $filter) {
            if ($filter->value === 'mfaenabled') {
                $mfa_filter = $filter;
                break;
            }
        }
        $this->assertNotNull($mfa_filter);

        $mfa_filter->set_data(['value' => 0]);
        list($sql, $params, $cache) = $report->build_query(false, true);

        $records = $DB->get_records_sql($sql, $params);
        $this->assertCount(3, $records);

        $names = [];
        foreach ($records as $record) {
            $names[$record->user_username] = $record->user_mfaenabled;
        }
        $expected = [
            'guest' => 0,
            'admin' => 0,
            'nomfa' => 0,
        ];
        $this->assertEqualsCanonicalizing($expected, $names);

        $mfa_filter->set_data(['value' => 1]);
        list($sql, $params, $cache) = $report->build_query(false, true);

        $records = $DB->get_records_sql($sql, $params);
        $this->assertCount(1, $records);
        $record = reset($records);
        $this->assertEquals('hasmfa', $record->user_username);
        $this->assertEquals('1', $record->user_mfaenabled);
    }
}
