<?php
/*
 * This file is part of Totara LMS
 *
 * Copyright (C) 2019 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Kian Nguyen <kian.nguyen@totaralearning.com>
 * @package totara_catalog
 */

defined('MOODLE_INTERNAL') || die();

use core_course\totara_catalog\course;
use totara_catalog\form\config_general;
use totara_catalog\local\config;
use totara_catalog\local\filter_handler;
use totara_catalog\catalog_retrieval;
use totara_program\totara_catalog\program;
use totara_catalog\filter;

class totara_catalog_wildcard_search_test extends \core_phpunit\testcase {
    /**
     * Test suite for searching the learning item from fts wildcard search. And this test suite is for
     * course only, plus that with wildcard search, the query should be un-accented by default.
     *
     * @return void
     */
    public function test_wildcard_search_with_single_word_for_course(): void {
        $gen = static::getDataGenerator();

        $course1 = $gen->create_course(
            [
                'shortname' => 'this is shortname',
                'fullname' => 'this is an actsman',
                'summary' => 'bolobala balabolo lpth p'
            ],
            ['createsections' => true]
        );

        $course2 = $gen->create_course(
            [
                'shortname' => 'tia course2',
                'fullname' => 'this is course2',
                'summary' => 'ajdiw klobp[ve epl ve;[be; \ b\lbp[r pojmf ioep lkp0 sh0fjib'
            ]
        );

        $terms = [
            [
                'value' => 'short*',
                'count' => 1,
                'course' => $course1->id
            ],
            [
                'value' => 'bolo*',
                'count' => 1,
                'course' => $course1->id
            ],
            [
                'value' => 'ac*',
                'count' => 1,
                'course' => $course1->id
            ],
            [
                'value' => 'helloworld',
                'count' => 0,
                'course' => null
            ],
            [
                'value' => 'sh0*',
                'count' => 1,
                'course' => $course2->id
            ]
        ];

        if ($this->check_index_process_completed()) {
            foreach ($terms as $term) {
                $filterhandler = filter_handler::instance();
                $filterhandler->reset_cache();

                $filter = $filterhandler->get_full_text_search_filter();

                // Let the php reference pointer doing this update $filterhandler for us.
                $filter->selector->set_current_data(
                    ['catalog_fts' => $term['value']]
                );

                $filter->datafilter->set_current_data(
                    $filter->selector->get_data()
                );

                $catalog = new catalog_retrieval();
                $page = $catalog->get_page_of_objects(20, 0);

                if (!property_exists($page, 'objects')) {
                    static::fail("No property 'objects' defined in \$page");
                }

                static::assertCount(
                    $term['count'],
                    $page->objects,
                    "Expecting the result with the pattern as '{$term['value']}' to equal '{$term['count']}'"
                );

                if (0 == $term['count']) {
                    continue;
                }

                $item = reset($page->objects);

                static::assertEquals($term['course'], $item->objectid);
                static::assertEquals(course::get_object_type(), $item->objecttype);
            }

            return;
        }

        static::fail("FTS index not completed in time, this is an issue with MSSQL");
    }

    /**
     * Test suite of checking the wildcard filter search for program.
     *
     * @return void
     */
    public function test_wildcard_search_with_single_word_for_program(): void {
        $gen = static::getDataGenerator();

        /** @var \totara_program\testing\generator $proggen */
        $proggen = $gen->get_plugin_generator('totara_program');
        $prog1 = $proggen->create_program(
            [
                'fullname' => 'This is actsman',
                'shortname' => 'This is shortname',
            ]
        );

        // Program 2, but it is not using in this test. Just in place to make sure that the query does
        // not include it.
        $proggen->create_program(
            [
                'fullname' => 'this is program2',
                'shortname' => 'tia program2'
            ]
        );

        $terms = [
            [
                'value' => 'ac*',
                'prog' => $prog1->id,
                'count' => 1
            ],
            [
                'value' => 'sho*',
                'prog' => $prog1->id,
                'count' => 1
            ],
            [
                // Will not default to a fallback like % search
                'value' => 'short',
                'prog' => null,
                'count' => 0
            ]
        ];

        if ($this->check_index_process_completed()) {
            foreach ($terms as $term) {
                $filterhandler = filter_handler::instance();
                $filterhandler->reset_cache();

                $filter = $filterhandler->get_full_text_search_filter();

                // Let the php reference pointer doing this update $filterhandler for us.
                $filter->selector->set_current_data(
                    ['catalog_fts' => $term['value']]
                );

                $filter->datafilter->set_current_data(
                    $filter->selector->get_data()
                );

                $catalog = new catalog_retrieval();
                $page = $catalog->get_page_of_objects(20, 0);

                if (!property_exists($page, 'objects')) {
                    static::fail("The object \$page does not have property 'objects'");
                }

                static::assertCount($term['count'], $page->objects);
                if (0 == $term['count']) {
                    continue;
                }

                $record = reset($page->objects);
                static::assertEquals($term['prog'], $record->objectid);
                static::assertEquals(program::get_object_type(), $record->objecttype);
            }

            return;
        }

        static::fail("FTS index not completed in time, this is an issue with MSSQL");
    }

    public function test_for_mysql_wildcard_search(): void {
        global $DB;
        $DB->delete_records('catalog');
        $this->resetAfterTest(true);

        // Creating courses indirectly updates the catalog.
        $courses = [
            [
                'shortname' => 'IND-22 Sales',
                'fullname' => 'this is an actsman1',
                'summary' => 'bolobala balabolo lpth p1'
            ],
            [
                'shortname' => '22 (Sep) Sharing',
                'fullname' => 'this is an actsman2',
                'summary' => 'bolobala balabolo lpth p2'
            ],
            [
                'shortname' => '22Sep) Sharing',
                'fullname' => 'this is an actsman3',
                'summary' => 'bolobala balabolo lpth p3'
            ]
        ];
        $terms_info = [
            '22Sep*' => 'positive',
            'ind-*' => 'positive',
            '22*' => 'positive',
            '*' => 'negative' // should return 0 results, cannot have a wildcard character in first index of a string for binary mode.
        ];

        $gen = static::getDataGenerator();
        foreach ($courses as $course) {
            $gen->create_course($course, ['createsections' => true]);
        }

        if ($this->check_index_process_completed()) {
            foreach ($terms_info as $term => $expected_result) {
                $filter_handler = filter_handler::instance();
                $filter_handler->reset_cache();
                $filter = $filter_handler->get_full_text_search_filter();

                // Let the php reference pointer doing this update $filterhandler for us.
                $filter->selector->set_current_data(
                    ['catalog_fts' => $term]
                );
                $filter->datafilter->set_current_data(
                    $filter->selector->get_data()
                );

                $catalog = new catalog_retrieval();
                $result = $catalog->get_page_of_objects(10, 0, -1, 'featured');

                // We are mainly just checking that no error happened with a wildcard or other character.
                if ($expected_result === 'positive') {
                    $this->assertNotEmpty($result->objects);
                } else {
                    $this->assertEmpty($result->objects);
                }
            }
        }
    }

    /**
     * Test suite for searching the learning item from partial word search.
     * @return void
     */
    public function test_partial_word_search(): void {
        global $DB;

        $gen = static::getDataGenerator();

        $gen->create_course(
            [
                'shortname' => 'l101',
                'fullname' => 'Landing with a plane 101',
                'summary' => 'A summary of a course'
            ],
            ['createsections' => true]
        );

        $gen->create_course(
            [
                'shortname' => 't101',
                'fullname' => 'Taking off with a plane 101',
                'summary' => 'A summary of a course'
            ]
        );

        $gen->create_course(
            [
                'shortname' => 'aero',
                'fullname' => 'Aerodrome operation',
                'summary' => 'Aerodrome operations. With the word plane in the ftsmedium'
            ]
        );

        $searches = [
            [config_general::SEARCH_FALLBACK_NONE, 'pl', 0],
            [config_general::SEARCH_FALLBACK_NONE, 'pla', 0],
            [config_general::SEARCH_FALLBACK_NONE, '*pl*', 0],
            [config_general::SEARCH_FALLBACK_NONE, 'taking', 1],
            [config_general::SEARCH_FALLBACK_NONE, 'plane', 3],

            [config_general::SEARCH_FALLBACK_NO_RESULTS, 'pla', 2],
            [config_general::SEARCH_FALLBACK_NO_RESULTS, '*pl*', 0],
            [config_general::SEARCH_FALLBACK_NO_RESULTS, 'taking', 1],
            [config_general::SEARCH_FALLBACK_NO_RESULTS, 'plane', 3],

            [config_general::SEARCH_FALLBACK_ALWAYS, 'pla', 2],
            [config_general::SEARCH_FALLBACK_ALWAYS, 'plane', 3],
            [config_general::SEARCH_FALLBACK_ALWAYS, '*pl*', 2],
            [config_general::SEARCH_FALLBACK_ALWAYS, 'wor', 0],
            [config_general::SEARCH_FALLBACK_ALWAYS, 'word', 1],
            [config_general::SEARCH_FALLBACK_ALWAYS, 'something', 0],
        ];

        // Not all databases extrapolate take with taking
        if (in_array($DB->get_dbfamily(), ['postgres', 'mssql'])) {
            $searches[] = [config_general::SEARCH_FALLBACK_NONE, 'take', 1];
            $searches[] = [config_general::SEARCH_FALLBACK_NO_RESULTS, 'take', 1];
            $searches[] = [config_general::SEARCH_FALLBACK_ALWAYS, 'take', 1];
        }
        // The build server's SQL server doesn't seem to accept * partial searches, these tests work locally
        if ($DB->get_dbfamily() != 'mssql') {
            $searches[] = [config_general::SEARCH_FALLBACK_NO_RESULTS, 'pl*', 3];
            $searches[] = [config_general::SEARCH_FALLBACK_ALWAYS, 'pl*', 3];
        }
        $config = config::instance();
        if ($this->check_index_process_completed()) {
            foreach ($searches as $currentsearch) {
                $fallbackmode = $currentsearch[0];
                $search = $currentsearch[1];
                $expectedresults = $currentsearch[2];
                $config->update(['search_fallback' => $fallbackmode]);

                $filterhandler = filter_handler::instance();
                $filterhandler->reset_cache();

                $filter = $filterhandler->get_full_text_search_filter();

                // Let the php reference pointer doing this update $filterhandler for us.
                $filter->selector->set_current_data(
                    ['catalog_fts' => $search]
                );

                $filter->datafilter->set_current_data(
                    $filter->selector->get_data()
                );

                $catalog = new catalog_retrieval();
                $page = $catalog->get_page_of_objects(20, 0);

                if (!property_exists($page, 'objects')) {
                    static::fail("No property 'objects' defined in \$page");
                }

                static::assertCount(
                    $expectedresults,
                    $page->objects,
                    "Expecting the result with the pattern as '{$search}' to find {$expectedresults} results with fallback mode: {$fallbackmode}"
                );
            }
        } else {
            static::fail("FTS index not completed in time, this is an issue with MSSQL");
        }
    }

    /**
     * When a record is inserted, the way that mssql work on it is that it does not wait for the process of
     * changing/populating the index to return the result back to PHP processor. But instead, it just return the
     * result of inserting records for php only. Therefore, we need to assure that the processes of fts
     * population are completely done, so that we can start performing test.
     *
     * @see https://docs.microsoft.com/en-us/sql/t-sql/functions/objectpropertyex-transact-sql?view=sql-server-2017
     * @return bool
     */
    private function check_index_process_completed(): bool {
        global $DB;
        if ('mssql' === $DB->get_dbvendor()) {
            $running = true;
            $attempted = 0;
            $sql = "SELECT OBJECTPROPERTYEX(OBJECT_ID(N'{$DB->get_prefix()}catalog'), N'TableFullTextPopulateStatus') as status;";

            while ($running) {
                $fts_indexing = $DB->get_record_sql($sql);
                switch ($fts_indexing->status) {
                    case '0':
                        // Idle, we're good to go.
                        $running = false;
                        break;
                    case '5':
                    case '6':
                        // Throttled, paused, or broken.
                        static::fail("FTS index cannot be completed due to an MSSQL error: TableFullTextPopulateStatus=" . $fts_indexing->status);
                        break;
                    default:
                        // Still running - Give it a several attempts,  if it exceeds 10 then MsSQL is having trouble.
                        if (10 < $attempted) {
                            return false;
                        }
                        $this->waitForSecond();
                        $attempted += 1;
                }
            }
        }

        return true;
    }
}