<?php
/*
 * This file is part of Totara LMS
 *
 * Copyright (C) 2016 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Sam Hemelryk <sam.hemelryk@totaralearning.com>
 * @author Alastair Munro <alastair.munro@totaralearning.com>
 * @package totara_core
 * @category user_learning
 */

namespace totara_core\user_learning;

use totara_core\data_provider\provider;

/**
 * User Learning Item base class.
 *
 * All other user learning item classes the represent in product learning requirements must extend this class.
 *
 * @package totara_core
 * @category user_learning
 */
#[\AllowDynamicProperties]
abstract class item_base implements item, designation {

    /**
     * The user this user learning item relates to.
     * @var \stdClass
     */
    public $user;

    /**
     * The id of this user learning ite,
     * @var int
     */
    public $id;
    public $shortname;
    public $fullname;
    public $description;
    public $description_format = null;
    public $image;

    public $url_view;
    public $progress;
    public $duedate;
    public $timecompleted;
    public $viewable;

    /**
     * Composite between type and id
     *
     * @var string
     */
    protected $unique_id;

    /**
     * The learning item record passed into the contructor.
     *
     * Kept as protected in case child classes require the entire object.
     * Noting they should not trust it!
     *
     * @var \stdClass
     */
    protected $learningitemrecord;

    protected $progress_pbar;

    private $context;
    private $owner;

    /**
     * Constructs a new user learning item instance given the user it relates to and the learning item.
     *
     * @param \stdClass|int $userorid
     * @param \stdClass $learningitemrecord
     */
    protected final function __construct($userorid, \stdClass $learningitemrecord) {
        $this->user = $this->resolve_user($userorid);
        $this->learningitemrecord = $learningitemrecord;
        $this->map_learning_item_record_data($learningitemrecord);
        $this->ensure_required_data_loaded();
    }

    /**
     * Ensure we have been able to load all the data required for the user's learning item.
     *
     * The {@see map_learning_item_record_data()} method must set at least the following properties:
     *   - id
     *   - shortname
     *   - fullname
     *   - url_view
     */
    protected function ensure_required_data_loaded() {
        $required = [
            'id',
            'shortname',
            'fullname',
            'url_view'
        ];
        foreach ($required as $field) {
            if ($this->$field === null) {
                // If you get here then please review your classes map_learning_item_record_data method and ensure it is setting all required fields.
                throw new \coding_exception('Method '.__CLASS__.'::map_learning_item_record_data() failed to load "'.$field.'" for the user learning item');
            }
        }
    }

    /**
     * Maps the from a learning item's data into the user learning item instance.
     *
     * This method must set at least the following properties:
     *   - id
     *   - shortname
     *   - fullname
     *   - url_view
     *
     * @param \stdClass $learningitemrecord The database record containing data that belongs to this user learning item.
     */
    abstract protected function map_learning_item_record_data(\stdClass $learningitemrecord);

    /**
     * Get the context for the item
     *
     * @return \context The context level for the item.
     */
    public function get_context() {
        if ($this->context === null) {
            /** @var \context $contextclass */
            $contextclass = \context_helper::get_class_for_level(static::get_context_level());
            if ($contextclass == 'context_system') {
                $this->context = \context_system::instance();
            } else {
                $this->context = $contextclass::instance($this->id);
            }
        }
        return $this->context;
    }

    /**
     * Sets the owner of an item.
     *
     * @throws \coding_exception if a circular reference has been encountered.
     * @param item_base $owner The parent of an item.
     */
    public function set_owner(item_base $owner) {
        if ($owner->has_owner() && $owner->get_owner() === $this) {
            throw new \coding_exception('Circular reference detected in user learning item ownership.');
        }
        $this->owner = $owner;
    }

    /**
     * Returns true if this user learning item has an owner.
     *
     * @return bool
     */
    public function has_owner() {
        return ($this->owner !== null);
    }

    /**
     * Gets the owner item of the learning item.
     *
     * @throws \coding_exception if the user learning item has no owner.
     * @return item_base The parent item.
     */
    public function get_owner() {
        if ($this->owner === null) {
            throw new \coding_exception('Attempting to get the owner of a user learning item that has no owner.');
        }
        return $this->owner;
    }

    /**
     * Returns the user record from the database given an id or the record.
     *
     * @param \stdClass|int $userorid
     * @return \stdClass
     */
    protected final static function resolve_user($userorid) {
        global $DB, $USER;
        if (is_object($userorid) && isset($userorid->id)) {
            $user = $userorid;
        } else if ($userorid == $USER->id) {
            $user = $USER;
        } else {
            $user = $DB->get_record('user', ['id' => (int)$userorid], '*', MUST_EXIST);
        }
        return $user;
    }


    /**
     * Exports data for rendering via a template.
     *
     * @return \stdClass
     */
    public function export_for_template() {
        $context = $this->get_context();
        $formatoptions = array('context' => $context);

        $record = new \stdClass;
        $record->component = $this->get_component();
        $record->component_name = $this->get_component_name();
        $record->type = $this->get_type();
        $record->id = $this->id;
        $record->shortname = format_string($this->shortname, true, $formatoptions);
        if ($this->shortname === $this->fullname) {
            $record->fullname = $record->shortname;
        } else {
            $record->fullname = format_string($this->fullname, true, $formatoptions);
        }
        if ($this->description !== null) {
            if ($this->description_format === null) {
                $record->description = format_string($this->description, true, $formatoptions);
            } else {
                $component  = ($record->component == 'totara_certification') ? 'totara_program' : $record->component;
                $description = file_rewrite_pluginfile_urls($this->description, 'pluginfile.php',
                    $context->id, $component, 'summary', 0);

                $record->description = format_text($description, $this->description_format, $formatoptions);
            }
        }
        $record->image = (string)$this->image;
        $record->url_view = (string)$this->url_view;

        if ($this instanceof item_has_progress) {
            $record->progress = $this->export_progress_for_template();
        }

        if ($this instanceof item_has_dueinfo) {
            $record->dueinfo = $this->export_dueinfo_for_template();
        }

        $record->viewable = $this->viewable;
        return $record;
    }

    /**
     * @inheritDoc
     */
    public function get_unique_id(): string {
        return $this->id.'-'.$this->get_type();
    }

    /**
     * @param $user_or_id
     * @param \stdClass $learning_item_record
     *
     * @return static
     */
    public static function create($user_or_id, \stdClass $learning_item_record): self {
        return new static($user_or_id, $learning_item_record);
    }

    /**
     * @inheritDoc
     */
    public static function get_data_provider(): ?provider {
        return null;
    }

    /**
     * Make sure site mobile feature is enabled to get downloadable learning item by mobile APP before calling this function.
     *
     * @return bool
     */
    public function get_download_friendly(): bool {
        return false;
    }
}
