<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Scott Davies <scott.davies@totara.com>
 * @package totara_hiearchy
 */

namespace totara_hierarchy\rb\display;

use totara_reportbuilder\rb\display\base;

/**
 * Display class intended for goal name.
 *
 * The goal name is displayed if the viewer has the 'totara/hierarchy:viewallgoals' capability, or if the goal is
 * their own, they have the 'totara/hierarchy:viewownpersonalgoal' capability or they meet the checks in the
 * goal 'get_permissions' method.
 *
 * @author Simon Player <simon.player@totaralearning.com>
 * @author Scott Davies <scott.davies@totara.com>
 * @package totara_hierarchy
 */
class goal_name_permissions_applied extends base {
    /**
     * Handles the display
     *
     * @param string $value
     * @param string $format
     * @param \stdClass $row
     * @param \rb_column $column
     * @param \reportbuilder $report
     * @return string
     */
    public static function display($value, $format, \stdClass $row, \rb_column $column, \reportbuilder $report) {
        global $CFG;

        if (empty($value)) {
            return '';
        }

        $extrafields = static::get_extrafields_row($row, $column);
        static $viewallgoalscap = null;
        if ($viewallgoalscap === null || PHPUNIT_TEST) {
            $viewallgoalscap = has_capability('totara/hierarchy:viewallgoals', \context_system::instance());
        }
        if ($viewallgoalscap) {
            return \totara_reportbuilder\rb\display\format_string::display($value, $format, $row, $column, $report);
        }

        include_once $CFG->dirroot . '/totara/hierarchy/prefix/goal/lib.php';

        $goal_id = $row->id;
        if ($extrafields->scope === 'personal' || $extrafields->scope == \goal::SCOPE_PERSONAL) {
            // Check the goal exists in the given scope.
            if (\goal::get_goal_item(['id' => $goal_id], \goal::SCOPE_PERSONAL)) {
                // Check the current user has permissions to see the goal of the goal user.
                $goal = new \goal();
                $permissions = $goal->get_permissions(null, $extrafields->userid);
                if ($permissions['can_view_personal'] || $permissions['can_edit_personal']) {
                    return \totara_reportbuilder\rb\display\format_string::display($value, $format, $row, $column, $report);
                }
            }
        } elseif ($extrafields->scope === 'company' || $extrafields->scope == \goal::SCOPE_COMPANY) {
            // Check the goal exists in the given scope.
            if (\goal::get_goal_item(['goalid' => $goal_id, 'userid' => $extrafields->userid], \goal::SCOPE_COMPANY)) {
                $goal = new \goal();
                $permissions = $goal->get_permissions(null, $extrafields->userid);
                if ($permissions['can_view_company'] || $permissions['can_edit_company']) {
                    return \totara_reportbuilder\rb\display\format_string::display($value, $format, $row, $column, $report);
                }
            }
        }

        return get_string('goalnamehidden', 'totara_hierarchy');
    }

    /**
     * Is this column graphable?
     *
     * @param \rb_column $column
     * @param \rb_column_option $option
     * @param \reportbuilder $report
     * @return bool
     */
    public static function is_graphable(\rb_column $column, \rb_column_option $option, \reportbuilder $report) {
        return false;
    }
}
