<?php
/**
 * This file is part of Totara Core
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Cody Finegan <cody.finegan@totara.com>
 * @package totara_message
 */

use core_phpunit\testcase;

/**
 * Testing that the task/alert system can run even if messaging is disabled
 */
class totara_message_message_disable_test extends testcase {
    /**
     * Testing tm_task_send method
     *
     * @return void
     */
    public function test_task_is_sent(): void {
        global $CFG;

        $user1 = $this->getDataGenerator()->create_user();
        $user2 = $this->getDataGenerator()->create_user();

        // Normal messaging enabled
        $CFG->messaging = true;
        $CFG->revert_TL_46666_until_T20 = false;

        $event = new stdClass;
        $event->userfrom = $user2;
        $event->userto = $user1;
        $event->contexturl = $CFG->wwwroot . '/';
        $event->icon = 'program-approve';
        $event->subject = 'Some task';
        $event->fullmessage = 'Full task message';
        $event->fullmessagehtml = '<div style="color:red">Full task message</div>';

        $sink = $this->redirectMessages();
        tm_task_send(clone($event));
        $messages = $sink->get_messages();
        $this->assertCount(1, $messages);
        $this->assertSame('Full task message', $messages[0]->fullmessage);

        // Messaging disabled, new behaviour
        $CFG->messaging = false;
        $CFG->revert_TL_46666_until_T20 = false;

        $sink = $this->redirectMessages();
        tm_task_send(clone($event));
        $messages = $sink->get_messages();
        $this->assertCount(1, $messages);
        $this->assertSame('Full task message', $messages[0]->fullmessage);

        // Messaging disabled, old behaviour
        $CFG->messaging = false;
        $CFG->revert_TL_46666_until_T20 = true;

        $sink = $this->redirectMessages();
        tm_task_send(clone($event));
        $messages = $sink->get_messages();
        $this->assertCount(0, $messages);
    }

    /**
     * Testing tm_alert_send method
     *
     * @return void
     */
    public function test_alert_is_sent(): void {
        global $CFG;

        $user1 = $this->getDataGenerator()->create_user();
        $user2 = $this->getDataGenerator()->create_user();

        // Normal messaging enabled
        $CFG->messaging = true;
        $CFG->revert_TL_46666_until_T20 = false;

        $event = new stdClass;
        $event->userfrom = $user1;
        $event->userto = $user2;
        $event->contexturl = $CFG->wwwroot . '/';
        $event->icon = 'program-approve';
        $event->subject = 'Some alert';
        $event->fullmessage = 'Full alert message';
        $event->fullmessagehtml = '<div style="color:red">Full alert message</div>';

        $sink = $this->redirectMessages();
        tm_alert_send(clone($event));
        $messages = $sink->get_messages();
        $this->assertCount(1, $messages);
        $this->assertSame('Full alert message', $messages[0]->fullmessage);

        // Messaging disabled, new behaviour
        $CFG->messaging = false;
        $CFG->revert_TL_46666_until_T20 = false;

        $sink = $this->redirectMessages();
        tm_alert_send(clone($event));
        $messages = $sink->get_messages();
        $this->assertCount(1, $messages);
        $this->assertSame('Full alert message', $messages[0]->fullmessage);

        // Messaging disabled, old behaviour
        $CFG->messaging = false;
        $CFG->revert_TL_46666_until_T20 = true;

        $sink = $this->redirectMessages();
        tm_alert_send(clone($event));
        $messages = $sink->get_messages();
        $this->assertCount(0, $messages);
    }

    /**
     * Testing send instant messages between users
     *
     * @return void
     */
    public function test_message_is_sent(): void {
        global $CFG;

        require_once $CFG->dirroot . '/message/externallib.php';

        $user1 = $this->getDataGenerator()->create_user();
        $user2 = $this->getDataGenerator()->create_user();

        $this->setUser($user1);

        // Normal messaging enabled
        $CFG->messaging = true;
        $CFG->revert_TL_46666_until_T20 = false;

        $sink = $this->redirectMessages();
        core_message_external::send_instant_messages([
            [
                'touserid' => $user2->id,
                'text' => 'Test message from user 1',
                'textformat' => FORMAT_PLAIN,
            ]
        ]);
        $messages = $sink->get_messages();
        $this->assertCount(1, $messages);
        $this->assertSame('Test message from user 1', $messages[0]->fullmessage);

        // Messaging disabled, new behaviour
        $CFG->messaging = false;
        $CFG->revert_TL_46666_until_T20 = false;

        $sink = $this->redirectMessages();
        $exception_message = '';
        try {
            core_message_external::send_instant_messages([
                [
                    'touserid' => $user2->id,
                    'text' => 'Test message from user 1',
                    'textformat' => FORMAT_PLAIN,
                ]
            ]);
        } catch (moodle_exception $ex) {
            $exception_message = $ex->getMessage();
        }
        $this->assertSame('Messaging is disabled on this site', $exception_message);
        $messages = $sink->get_messages();
        $this->assertCount(0, $messages);

        // Messaging disabled, old behaviour
        $CFG->messaging = false;
        $CFG->revert_TL_46666_until_T20 = true;

        $sink = $this->redirectMessages();
        $exception_message = '';
        try {
            core_message_external::send_instant_messages([
                [
                    'touserid' => $user2->id,
                    'text' => 'Test message from user 1',
                    'textformat' => FORMAT_PLAIN,
                ]
            ]);
        } catch (moodle_exception $ex) {
            $exception_message = $ex->getMessage();
        }
        $this->assertSame('Messaging is disabled on this site', $exception_message);
        $messages = $sink->get_messages();
        $this->assertCount(0, $messages);
    }
}