<?php
/*
 * This file is part of Totara LMS
 *
 * Copyright (C) 2010 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Rob Tyler <rob.tyler@totaralms.com>
 * @package totara_plan
 */

defined('MOODLE_INTERNAL') || die();

use totara_program\assignments\assignments;
use totara_program\program;
use totara_competency\entity\competency_achievement;
use totara_job\job_assignment;
use totara_competency\entity\assignment;

class totara_plan_lib_test extends \core_phpunit\testcase {
    /** @var \totara_plan\testing\generator $plangenerator */
    protected $plangenerator;

    protected function tearDown(): void {
        $this->plangenerator = null;
        parent::tearDown();
    }

    protected function setUp(): void {
        global $CFG;
        parent::setUp();

        require_once($CFG->dirroot.'/totara/plan/lib.php');
        require_once($CFG->dirroot.'/totara/hierarchy/lib.php');

        $this->setAdminUser();

        $this->plangenerator = $this->getDataGenerator()->get_plugin_generator('totara_plan');
    }

    /**
     * Test creating a learning plan and adding a course.
     */
    public function test_add_course_to_learning_plan() {

        // Create a learning plan.
        $plan = $this->plangenerator->create_learning_plan();

        $course = $this->getDataGenerator()->create_course();

        // Add the course to the learning plan.
        $this->add_component_to_learning_plan($plan->id, 'course', $course->id);
    }

    /**
     * Test creating a learning plan and adding a competency.
     */
    public function test_add_competency_to_learning_plan() {
        global $DB;

        // Create a learning plan.
        $plan = $this->plangenerator->create_learning_plan();

        // Get a hierarchy generator.
        /** @var \totara_hierarchy\testing\generator $hierarchygenerator */
        $hierarchy_gen = $this->getDataGenerator()->get_plugin_generator('totara_hierarchy');

        // Create a new competency framework and check it exists.
        $comp_fw = $hierarchy_gen->create_framework('competency');
        $exists = $DB->record_exists('comp_framework', array('id' => $comp_fw->id));
        // Assert the existence of the framework.
        $this->assertTrue($exists);

        // Create a new competency and check it exists.
        $comp = $hierarchy_gen->create_hierarchy($comp_fw->id, 'competency', array('fullname' => 'Test Competency'));
        $exists = $DB->record_exists('comp', array('id' => $comp->id));
        // Assert the existence of the competency.
        $this->assertTrue($exists);

        $this->add_component_to_learning_plan($plan->id, 'competency', $comp->id);
    }

    /**
     * Test creating a learning plan and adding a program.
     */
    public function test_add_program_to_learning_plan() {
        global $DB;

        // Create a learning plan.
        $plan = $this->plangenerator->create_learning_plan();

        /** @var \totara_plan\testing\generator $plangenerator */
        $program_gen = $this->getDataGenerator()->get_plugin_generator('totara_program');

        // Create a new program and check it exists
        $program = $program_gen->create_program();
        $exists = $DB->record_exists('prog', array('id' => $program->id));
        $this->assertTrue($exists);
        // Add program to learning plan.
        $this->add_component_to_learning_plan($plan->id, 'program', $program->id);
    }

    /**
     * Add a component to a competency framework.
     * @param int $planid
     * @param string $component
     * @param int $componentid
     */
    protected function add_component_to_learning_plan($planid, $component, $componentid) {
        global $DB;
        // Add the competency to the learning plan.
        $plan = new development_plan($planid);
        $componentobj = $plan->get_component($component);
        $componentobj->update_assigned_items(array($componentid));
        // Check the course has been assigned to the learning plan.
        $exists = $DB->record_exists('dp_plan_' . $component . '_assign', array('planid' => $planid, $component . 'id' => $componentid));
        // Assert the existence of the record.
        $this->assertTrue($exists);
    }

    /**
     * Change the state of all cert and prog completion records to certified, before the window opens.
     *
     * Borrowed from totara/certificaton/tests/certification_completion_test.php.
     */
    private function shift_completions_to_certified($timecompleted) {
        global $DB;

        // Manually change their state.
        $sql = "UPDATE {prog_completion}
                   SET status = :progstatus, timecompleted = :timecompleted, timedue = :timedue
                 WHERE coursesetid = 0";
        $params = array('progstatus' => program::STATUS_PROGRAM_COMPLETE, 'timecompleted' => $timecompleted,
            'timedue' => $timecompleted + 2000);
        $DB->execute($sql, $params);
        $sql = "UPDATE {certif_completion}
                   SET status = :certstatus, renewalstatus = :renewalstatus, certifpath = :certifpath,
                       timecompleted = :timecompleted, timewindowopens = :timewindowopens, timeexpires = :timeexpires";
        $params = array('certstatus' => CERTIFSTATUS_COMPLETED, 'renewalstatus' => CERTIFRENEWALSTATUS_NOTDUE,
            'certifpath' => CERTIFPATH_RECERT, 'timecompleted' => $timecompleted, 'timewindowopens' => $timecompleted + 1000,
            'timeexpires' => $timecompleted + 2000);
        $DB->execute($sql, $params);
    }

    /**
     * Tests the function dp_get_rol_tabs_visible.
     *
     * The particular scenario being tested here is that the programs is not returned as visible
     * when the user is only enrolled in certs.
     *
     * That issue can happen because we might test for program completion records when deciding whether
     * the programs tab should be visible. But certs use program completion records as well.
     */
    public function test_dp_get_rol_tabs_visible_cert_not_program() {
        $generator = $this->getDataGenerator();
        $user = $generator->create_user();

        /** @var \totara_program\testing\generator $program_generator */
        $program_generator = $generator->get_plugin_generator('totara_program');

        // This program won't be assigned. Just helps to detect things like joins being done incorrectly.
        $program =  $program_generator->create_program();

        // We'll have the user complete this certification.
        $certprogramid = $program_generator->create_certification();
        $certprogram = new program($certprogramid);

        $program_generator->assign_to_program($certprogram->id, assignments::ASSIGNTYPE_INDIVIDUAL, $user->id, null, true);

        $this->shift_completions_to_certified(time());
        $visible_tabs = dp_get_rol_tabs_visible($user->id);
        $this->assertContains('certifications', $visible_tabs);
        $this->assertNotContainsEquals('programs', $visible_tabs);
    }

    /**
     * Tests the function dp_get_rol_tabs_visible.
     *
     * The particular scenario is the inverse of the test above (test_dp_get_rol_tabs_visible_cert_not_program).
     * The issue tested above is the main concern. We're just making sure the opposite doesn't happen while
     * making the above test pass.
     */
    public function test_dp_get_rol_tabs_visible_program_not_cert() {
        $generator = $this->getDataGenerator();
        $user = $generator->create_user();

        /** @var \totara_program\testing\generator $program_generator */
        $program_generator = $generator->get_plugin_generator('totara_program');

        // We'll have the user complete this program.
        $program =  $program_generator->create_program();

        // Adding a certification. This won't be assigned though.
        $certprogramid = $program_generator->create_certification();
        $certprogram = new program($certprogramid);

        $program_generator->assign_to_program($program->id, assignments::ASSIGNTYPE_INDIVIDUAL, $user->id, null, true);

        $progcompletion = prog_load_completion($program->id, $user->id);
        $progcompletion->status = program::STATUS_PROGRAM_COMPLETE;
        $progcompletion->timestarted = time();
        $progcompletion->timecompleted = time();
        prog_write_completion($progcompletion);

        $visible_tabs = dp_get_rol_tabs_visible($user->id);
        $this->assertContains('programs', $visible_tabs);
        $this->assertNotContainsEquals('certifications', $visible_tabs);
    }

    /**
     * Tests the function dp_get_rol_tabs_visible for the competency tab.
     */
    public function test_dp_get_rol_tabs_visible_competency() {
        global $DB;

        $generator = $this->getDataGenerator();
        $user1 = $generator->create_user();
        self::setUser($user1);

        /** @var \totara_competency\testing\generator $competency_generator */
        $competency_generator = $generator->get_plugin_generator('totara_competency');
        $hierarchy_generator = $competency_generator->hierarchy_generator();

        $pfw = $hierarchy_generator->create_pos_frame(['fullname' => 'Pos Framework']);
        $pos = $hierarchy_generator->create_pos(['frameworkid' => $pfw->id, 'fullname' => 'Position 1']);

        $cfw = $hierarchy_generator->create_comp_frame([]);
        $competency = $competency_generator->create_competency(null, $cfw->id, [
            'shortname' => 'acc',
            'fullname' => 'Accounting',
        ]);

        $status = ['status' => assignment::STATUS_ACTIVE];
        $position_assignment = $competency_generator->assignment_generator()->create_position_assignment($competency->id, $pos->id, $status);

        $job_data = [
            'userid' => $user1->id,
            'idnumber' => 'dev13',
            'fullname' => 'Developer',
            'positionid' => $pos->id
        ];
        job_assignment::create($job_data);

        $comp_achievement = new stdClass();
        $comp_achievement->competency_id = $competency->id;
        $comp_achievement->user_id = $user1->id;
        $comp_achievement->assignment_id = $position_assignment->id;
        $comp_achievement->status = competency_achievement::ACTIVE_ASSIGNMENT;
        $comp_achievement->proficient = 0;
        $now = time();
        $comp_achievement->time_created = $now;
        $comp_achievement->time_proficient = $now;
        $comp_achievement->time_status = $now;
        $comp_achievement->time_scale_value = $now;
        $comp_achievement->last_aggregated = $now;
        $id = $DB->insert_record('totara_competency_achievement', $comp_achievement);

        $visible_tabs = dp_get_rol_tabs_visible($user1->id);
        self::assertEmpty($visible_tabs);

        // Revert config to true
        set_config('revert_TL_44715_until_t20', 1);
        $visible_tabs = dp_get_rol_tabs_visible($user1->id);
        $this->assertContainsEquals('competencies', $visible_tabs);

        // Revert config to false, but update scale_value_id.
        set_config('revert_TL_44715_until_t20', 0);
        $comp_achievement->id = $id;
        $comp_achievement->scale_value_id = 0;
        $DB->update_record('totara_competency_achievement', $comp_achievement);
        $visible_tabs = dp_get_rol_tabs_visible($user1->id);
        $this->assertContainsEquals('competencies', $visible_tabs);
    }
}
