<?php
/*
 * This file is part of Totara Learn
 *
 * Copyright (C) 2018 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Samantha Jayasinghe <samantha.jayasinghe@totaralearning.com>
 * @package totara_reportbuilder
 */

namespace totara_reportbuilder\rb\content;

/*
 * Restrict the report content by a particular audience
 */

use core\entity\cohort;
use core\orm\query\builder;
use core\record\tenant;

final class audience extends base {

    const TYPE = 'audience_content';

    /**
     * Generate the SQL to apply this content restriction
     *
     * @param string  $field    SQL field to apply the restriction against
     * @param integer $reportid ID of the report
     *
     * @return array containing SQL snippet to be used in a WHERE clause, as well as array of SQL params
     */
    public function sql_restriction($field, $reportid) {

        $settings = \reportbuilder::get_all_settings($reportid, self::TYPE);
        $cohortid = $settings['audience'] ?? 0;

        $restriction = " 1=1 ";
        $params = [];

        if (!empty($cohortid)) {
            $restriction = "({$field} IN (SELECT userid
                                       FROM {cohort_members} cm
                                       WHERE
                                         cm.cohortid = :audiencecontentaudienceid
                                     ))";
            $params = ['audiencecontentaudienceid' => $cohortid];
        }

        return [$restriction, $params];
    }

    /**
     * Generate a human-readable text string describing the restriction
     *
     * @param string  $title    Name of the field being restricted
     * @param integer $reportid ID of the report
     *
     * @return string Human readable description of the restriction
     */
    public function text_restriction($title, $reportid) {
        return get_string('reportbuilderaudience', 'totara_reportbuilder');
    }

    /**
     * Adds form elements required for this content restriction's settings page
     *
     * @param object  &$mform   Moodle form object to modify (passed by reference)
     * @param integer $reportid ID of the report being adjusted
     * @param string  $title    Name of the field the restriction is acting on
     */
    public function form_template(&$mform, $reportid, $title) {
        global $OUTPUT;

        $mform->addElement(
            'header', 'user_audience_header',
            get_string('reportbuilderaudience', 'totara_reportbuilder')
        );
        $mform->setExpanded('user_audience_header');

        $context = $this->get_report_context($reportid);
        if ($this->has_audience_capability($context)) {
            $cohortid = \reportbuilder::get_setting($reportid, self::TYPE, 'audience');
            $enable = \reportbuilder::get_setting($reportid, self::TYPE, 'enable');

            $mform->addElement(
                'checkbox', 'user_audience_enable', '',
                get_string('reportbuilderaudience', 'totara_reportbuilder')
            );
            $mform->setDefault('user_audience_enable', $enable);
            $mform->addElement(
                'select', 'user_audience',
                get_string('includerecordsfrom', 'totara_reportbuilder'),
                self::get_cohort_select_options($context->id)
            );
            $mform->setDefault('user_audience', $cohortid);

            $mform->disabledIf('user_audience', 'user_audience_enable', 'notchecked');
            $mform->disabledIf('user_audience_enable', 'contentenabled', 'eq', 0);
        } else {
            $settings = \reportbuilder::get_all_settings($reportid, self::TYPE);
            $cohortid = $settings['audience'] ?? 0;

            $warning = get_string('reportbuilderaudiencenopermission', 'totara_reportbuilder');

            if (!empty($cohortid)) {
                $warning = get_string('reportbuilderaudiencenopermissionapplied', 'totara_reportbuilder');
            }
            $warning = $OUTPUT->notification($warning, 'warning');
            $mform->addElement('html', $warning);
        }

        $mform->addHelpButton('user_audience_header', 'reportbuilderaudience', 'totara_reportbuilder');
    }

    /**
     * Processes the form elements created by {@link form_template()}
     *
     * @param integer $reportid ID of the report to process
     * @param object  $fromform Moodle form data received via form submission
     *
     * @return boolean True if form was successfully processed
     */
    public function form_process($reportid, $fromform) {
        $status = true;

        $context = $this->get_report_context($reportid);
        if ($this->has_audience_capability($context)) {
            $audience = $fromform->user_audience ?? 0;
            $audienceenable = $fromform->user_audience_enable ?? 0;

            if (!empty($audienceenable) && empty($audience)) {
                $link = new \moodle_url('/totara/reportbuilder/content.php', ['id' => $reportid]);
                throw new \moodle_exception('noaudienceselected', 'totara_reportbuilder', $link);
            }

            $status = $status && \reportbuilder::update_setting($reportid, self::TYPE, 'audience', $audience);
            $status = $status && \reportbuilder::update_setting($reportid, self::TYPE, 'enable', $audienceenable);
        }

        return $status;
    }

    /**
     * Get the report's context
     *
     * @param int $reportid
     * @return \context
     */
    private function get_report_context(int $reportid): \context {
        // Parent context for audiences options in report
        $report = builder::table('report_builder')->find($reportid);

        return empty($report->tenantid)
            ? \context_system::instance()
            : tenant::fetch($report->tenantid)->category_context;
    }

    /**
     * Check audience capability
     *
     * @param \context $context
     * @return bool
     */
    private function has_audience_capability(\context $context): bool {
        return has_capability("moodle/cohort:view", $context);
    }

    /**
     * Update default audience restrictions
     *
     * @param int $reportid
     * @param int $cohortid
     */
    public function set_default_restriction(int $reportid, int $cohortid) {
        global $DB;

        // update content mode to ANY criteria
        $report = new \stdClass();
        $report->id = $reportid;
        $report->contentmode = REPORT_BUILDER_CONTENT_MODE_ANY;
        $DB->update_record('report_builder', $report);

        // enable audience and update global settings
        \reportbuilder::update_setting($reportid, self::TYPE, 'audience', $cohortid);
        \reportbuilder::update_setting($reportid, self::TYPE, 'enable', 1);
    }

    /**
     * get all options for an audience/cohort select box
     *
     * @param int|null $context_id Parent context of audiences
     * @return array
     */
    public static function get_cohort_select_options(int $context_id = null): array {
        $builder = cohort::repository()->as('c')
            ->select_raw('c.id as id, c.name as name, c.idnumber as idnumber')
            ->order_by('name')
            ->order_by('idnumber');

        if (!empty($context_id)) {
            $builder->join(['context_map', 'cm'], function (builder $builder) use ($context_id) {
                $builder->where_raw('cm.childid = c.contextid')
                    ->where('cm.parentid', $context_id);
            });
        }
        $cohorts = $builder->get();
        $options = [];
        foreach ($cohorts as $cohort) {
            $options[$cohort->id] = format_string("{$cohort->name} ({$cohort->idnumber})");
        }

        return [ get_string('no_audience_defined', 'rb_source_user') ] + $options;
    }

}
