var SCORM_TRUE = 'true';
var SCORM_FALSE = 'false';
var SCORM_NO_ERROR = '0';

function ScormAPI() {
  this.initialized = false;
  this.terminated = false;
}

ScormAPI.prototype._checkValid = function () {
  if (!this.initialized) {
    throw new ScormError(10105, 'Instance has not been initialized');
  }
  if (this.terminated) {
    throw new ScormError(10104, 'Instance has been terminated');
  }
};

ScormAPI.prototype.initialize = function (window) {
  this.api = this._getAPI(window);

  if (this.api == null) {
    throw new ScormError(10102, 'Cannot establish connection to LMS');
  }

  var result = this.api.Initialize('');

  if (result == SCORM_FALSE) {
    throw ScormError.fromScormAPI(this.api);
  }

  this.initialized = true;
};

ScormAPI.prototype.terminate = function (e) {
  var result;

  this._checkValid();

  var result = this.api.Terminate('', e);

  this.terminated = true;

  if (result == SCORM_FALSE) {
    throw ScormError.fromScormAPI(this.api);
  }
};

ScormAPI.prototype.getValue = function (element) {
  var result;

  this._checkValid();

  result = this.api.GetValue(element);

  if (result == '') {
    ScormError.throwIfError(this.api);
  }

  return result;
};

ScormAPI.prototype.setValue = function (element, value) {
  var result;

  this._checkValid();

  result = this.api.SetValue(element, value);

  if (result == SCORM_FALSE) {
    throw ScormError.fromScormAPI(this.api);
  }
};

ScormAPI.prototype._scanForAPI = function (win) {
  var tries = 0;
  while (!win.API_1484_11 && win.parent && win.parent != win) {
    tries++;
    if (tries > 500) {
      return null;
    }
    win = win.parent;
  }
  return win.API_1484_11;
};

ScormAPI.prototype._getAPI = function (win) {
  var API;
  if (win.parent && win.parent != win) {
    API = this._scanForAPI(win.parent);
  }
  if (!API && win.opener) {
    API = this._scanForAPI(win.opener);
  }
  return API;
};

function ScormError(code, message) {
  var instance = new Error(code + ': ' + message);
  instance.name = 'ScormError';
  instance.code = code;
  Object.setPrototypeOf(instance, Object.getPrototypeOf(this));
  if (Error.captureStackTrace) {
    Error.captureStackTrace(instance, ScormError);
  }
  return instance;
}

ScormError.prototype = Object.create(Error.prototype, {
  constructor: {
    value: Error,
    enumerable: false,
    writable: true,
    configurable: true,
  },
});

Object.setPrototypeOf(ScormError, Error);

ScormError.fromScormAPI = function (API) {
  var code = API.GetLastError();
  var errorString = API.GetErrorString(code);
  var diagnostic = API.GetDiagnostic(code);

  return new ScormError(code, errorString + '\n' + diagnostic);
};

ScormError.throwIfError = function (API) {
  var errorNumber = API.GetLastError();

  if (errorNumber != SCORM_NO_ERROR) {
    throw ScormError.fromScormAPI(API);
  }
};
